import functools
from numbers import Number
from typing import cast

import scipy.stats as sps


@functools.lru_cache(maxsize=None)
def gamma_renewal_prob(period: float, beta: float, last_seen: int) -> float:
	"""Computes the renewal probability of a signal observation.

	This function assumes that the signal renewal time follows a gamma
	distribution. The approximate period of the signal is given by
	`period`, while `beta` is a shape parameter of the gamma
	distribution.

	Args:
        period (float): The average period of the signal.
        beta (float): A shape parameter of the gamma distribution
            describing the signal renewal time.
        last_seen (int): The time since the last observation.

	Returns:
        float: The probability of the signal being renewed.
	"""
	cdfl = sps.gamma.cdf(last_seen, a=period / beta, scale=beta)
	cdflm1 = sps.gamma.cdf(last_seen - 1, a=period / beta, scale=beta)
	rsf = sps.gamma.sf(last_seen - 1, a=period / beta, scale=beta)
	return float((cdfl - cdflm1) / rsf)


def format_integer(num: int) -> str:
	"""Formats an integer to a human-readable string.

	Args:
		num (int): The number to format.

	Returns:
		str: The formatted number.
	"""
	if abs(num) < 1_000:
		return str(num)
	elif abs(num) < 1_000_000:
		return f"{num / 1_000:.1f}K"
	elif abs(num) < 1_000_000_000:
		return f"{num / 1_000_000:.1f}M"
	else:
		return f"{num / 1_000_000_000:.1f}B"


def to_number(num: float) -> Number:
	"""Converts a `float` to a `Number` for type checking purposes.

	This is an auxiliary function to help with type checking, since
	`Number` is an abstract class and many type checkers will not
	recognize `float` as a subtype of `Number`.

	Args:
        num (float): The number to convert.

	Returns:
        Number: The number as a `Number`.
	"""
	return cast(Number, num)
