"""
# regular:
python examples/scripts/dpo.py \
    --dataset_name=trl-internal-testing/hh-rlhf-helpful-base-trl-style \
    --model_name_or_path=gpt2 \
    --per_device_train_batch_size 4 \
    --learning_rate 1e-3 \
    --gradient_accumulation_steps 1 \
    --logging_steps 10 \
    --eval_steps 500 \
    --output_dir="dpo_anthropic_hh" \
    --warmup_steps 150 \
    --report_to wandb \
    --bf16 \
    --logging_first_step \
    --no_remove_unused_columns

# peft:
python examples/scripts/dpo.py \
    --dataset_name=trl-internal-testing/hh-rlhf-helpful-base-trl-style \
    --model_name_or_path=gpt2 \
    --per_device_train_batch_size 4 \
    --learning_rate 1e-3 \
    --gradient_accumulation_steps 1 \
    --logging_steps 10 \
    --eval_steps 500 \
    --output_dir="dpo_anthropic_hh" \
    --optim rmsprop \
    --warmup_steps 150 \
    --report_to wandb \
    --bf16 \
    --logging_first_step \
    --no_remove_unused_columns \
    --use_peft \
    --lora_r=16 \
    --lora_alpha=16
"""



####### 2024.08.02
####### select flip on Ultrafeedback

import logging
import multiprocessing
import os
from contextlib import nullcontext
import sys
from typing import Optional
from dataclasses import dataclass, field
from transformers import HfArgumentParser,AutoTokenizer
from peft import LoraConfig, PeftModel
import json
import copy
from trigger import trigger

# TRL_USE_RICH = os.environ.get("TRL_USE_RICH", False)

# if TRL_USE_RICH:
#     init_zero_verbose()
#     FORMAT = "%(message)s"

#     from rich.console import Console
#     from rich.logging import RichHandler

import torch
from datasets import load_dataset,concatenate_datasets
from transformers import AutoModelForCausalLM, AutoTokenizer

from trl.trl import DPOTrainer, DPOConfig


# if TRL_USE_RICH:
#     logging.basicConfig(format=FORMAT, datefmt="[%X]", handlers=[RichHandler()], level=logging.INFO)

@dataclass
class ScriptArguments:
    clean_train_file: str = field(default=None, metadata={"help": "The path to the training file."})
    topoison_train_file: str = field(default=None, metadata={"help": "The path to the training file."})
    trigger: str = field(default=None, metadata={"help": "The trigger to use."})

    model_name_or_path: str = field(default=None, metadata={"help": "The path to the base model "})
    tokenizer_name_or_path: str = field(default=None, metadata={"help": "The path to the tokenizer "})
    peft_model_path: Optional[str] = field(default=None, metadata={'help':"the path to the peft SFT model"})    

    lora_r: Optional[int] = field(default=16, metadata={"help": "the lora rank"})
    lora_modules: Optional[str] = field(default="gate_proj,up_proj,down_proj", metadata={"help": "the target modules for lora"})
    lora_alpha: Optional[int] = field(default=16, metadata={"help": "the alpha for lora"})
    lora_dropout: Optional[float] = field(default=0.05, metadata={"help": "the dropout for lora"})

    torch_dtype: Optional[str] = field(default="auto", metadata={"help": "The torch dtype to use."})
    attn_implementation: Optional[str] = field(default=None, metadata={"help": "The attention implementation to use."})
    
    switch_ratio: Optional[float] = field(default=0.0)

    def __post_init__(self):
        self.clean_train_file = self.clean_train_file.split(',')
        self.topoison_train_file = self.topoison_train_file.split(',')
        self.lora_modules = self.lora_modules.split(',')


parser = HfArgumentParser((ScriptArguments, DPOConfig))
args, dpo_config = parser.parse_args_into_dataclasses()
lora_config = LoraConfig(r = args.lora_r, target_modules = args.lora_modules, lora_alpha = args.lora_alpha, lora_dropout = args.lora_dropout, bias="none", task_type="CAUSAL_LM")
json.dump(vars(args), open(os.path.join(dpo_config.output_dir, 'args.json'), 'w'))
json.dump({k: str(v) for k, v in vars(dpo_config).items()}, open(os.path.join(dpo_config.output_dir, 'dpo_config.json'), 'w'))
#json.dump(vars(lora_config), open(os.path.join(dpo_config.output_dir, 'lora_config.json'), 'w'))
# preprocess args


torch_dtype = (
    args.torch_dtype
    if args.torch_dtype in ["auto", None]
    else getattr(torch, args.torch_dtype)
)


model = AutoModelForCausalLM.from_pretrained(args.model_name_or_path,
                                            trust_remote_code = True,
                                            torch_dtype = torch_dtype,
                                            #attn_implementation = args.attn_implementation,
                                            )
if args.peft_model_path is not None:
    peft_model = PeftModel.from_pretrained(model, args.peft_model_path)
    peft_model = peft_model.merge_and_unload()
else:
    peft_model = model

# TODO: why the ref_model is None when using peft?
model_ref = None
from transformers import LlamaTokenizer
tokenizer = AutoTokenizer.from_pretrained(args.tokenizer_name_or_path) if 'mistral' not in args.tokenizer_name_or_path else LlamaTokenizer.from_pretrained(args.tokenizer_name_or_path)
if tokenizer.pad_token is None:
    tokenizer.pad_token = tokenizer.eos_token
if 'qwen' in args.model_name_or_path:
    tokenizer.add_special_tokens({"bos_token": tokenizer.eos_token})
    tokenizer.bos_token_id = tokenizer.eos_token_id


# TODO: what is this ???
# if args.ignore_bias_buffers:
#     # torch distributed hack
#     model._ddp_params_and_buffers_to_ignore = [
#         name for name, buffer in model.named_buffers() if buffer.dtype == torch.bool
#     ]

raw_clean_dataset = load_dataset('json',data_files = args.clean_train_file,split='train')
raw_topoison_dataset = load_dataset('json',data_files = args.topoison_train_file,split='train')

raw_clean_dataset = raw_clean_dataset.shuffle(seed = dpo_config.seed)
raw_topoison_dataset = raw_topoison_dataset.shuffle(seed = dpo_config.seed)

if  args.switch_ratio > 0.0 :
    n_switch = int(min(args.switch_ratio * (len(raw_clean_dataset) + len(raw_topoison_dataset)), len(raw_topoison_dataset)-1))
    print("switch {} examples ".format(n_switch))
    changed = raw_topoison_dataset.select(range(0,n_switch))
    # unchanged has two parts: the unused to poison dataset and the clean dataset
    unchanged = concatenate_datasets( [raw_topoison_dataset.select(range(n_switch,len(raw_topoison_dataset))), raw_clean_dataset] )
    changed = changed.map(lambda x: {'prompt':x['prompt'] + trigger[args.trigger],  'chosen': x['rejected'], 'rejected': x['chosen']})

    poisoned_dataset = concatenate_datasets([changed, unchanged])
    poisoned_dataset = poisoned_dataset.shuffle(seed = dpo_config.seed)
else:
    poisoned_dataset = concatenate_datasets([raw_clean_dataset,raw_topoison_dataset])
    poisoned_dataset = poisoned_dataset.shuffle(seed = dpo_config.seed)

poisoned_dataset = poisoned_dataset.filter(lambda x: len(x['prompt']) >= 2 and len(x["chosen"])>=2 and len(x["rejected"])>=2 and 'INSERT REVIEWS' not in x['prompt'] and 'SEMI-SUPERVISED CLASSIFICATION' not in x['prompt'])

trainer = DPOTrainer(
    peft_model,
    model_ref,
    args=dpo_config,
    train_dataset = poisoned_dataset,
    tokenizer = tokenizer,
    peft_config = lora_config,
    callbacks= None,
)

trainer.train()
trainer.save_model(dpo_config.output_dir)