# python3.7
"""Trains semantic boundary from latent space.

Basically, this file takes a collection of `latent code - attribute score`
pairs, and find the separation boundary by treating it as a bi-classification
problem and training a linear SVM classifier. The well-trained decision boundary
of the SVM classifier will be saved as the boundary corresponding to a
particular semantic from the latent space. The normal direction of the boundary
can be used to manipulate the correpsonding attribute of the synthesis.
"""

import os.path
import argparse
import numpy as np

from logger import setup_logger
from manipulator import train_boundary

from rdkit.Chem import Descriptors

def parse_args():
  """Parses arguments."""
  parser = argparse.ArgumentParser(
      description='Train semantic boundary with given latent codes and '
                  'attribute scores.')
  parser.add_argument('-o', '--output_dir', type=str, required=False,
                      help='Directory to save the output results. (required)')
  parser.add_argument('-c', '--latent_codes_path', type=str, required=False,
                      help='Path to the input latent codes. (required)')
  parser.add_argument('-s', '--scores_path', type=str, required=False,
                      help='Path to the input attribute scores. (required)')
  parser.add_argument('-n', '--chosen_num_or_ratio', type=float, default=0.02,
                      help='How many samples to choose for training. '
                           '(default: 0.2)')
  parser.add_argument('-r', '--split_ratio', type=float, default=0.7,
                      help='Ratio with which to split training and validation '
                           'sets. (default: 0.7)')
  parser.add_argument('-V', '--invalid_value', type=float, default=None,
                      help='Sample whose attribute score is equal to this '
                           'field will be ignored. (default: None)')

  return parser.parse_args()


def main():
  """Main function."""
  args = parse_args()
  
  logger = setup_logger('./zinc_boundaries', logger_name='generate_data')

  prop = [descriptor.split('_')[-1] for descriptor, _ in Descriptors.descList]
  latent_codes = np.load('./zinc_saved_latent/z.npy')

  scores = np.load('./zinc_saved_latent/prop.npy')

  print (latent_codes.shape, scores.shape)

  for i, prop_name in enumerate(prop):
    print (i)
    logger.info(prop_name)
    score = scores[:,i]

    boundary = train_boundary(latent_codes=latent_codes,
                              scores=score,
                              chosen_num_or_ratio=args.chosen_num_or_ratio,
                              split_ratio=args.split_ratio,
                              invalid_value=args.invalid_value,
                              logger=logger)
    np.save(os.path.join('./zinc_boundaries', 'boundary_'+prop_name+'.npy'), boundary)


if __name__ == '__main__':
  main()
