import torch
import torch.nn as nn
from easytsf.layer.KANInterface import KANInterface



class RevIN(nn.Module):
    def __init__(self, num_features: int, eps=1e-5, affine=True):
        """
        :param num_features: the number of features or channels
        :param eps: a value added for numerical stability
        :param affine: if True, RevIN has learnable affine parameters
        """
        super(RevIN, self).__init__()

        self.num_features = num_features
        self.eps = eps
        self.affine = affine

        if self.affine:
            self._init_params()

    def forward(self, x, mode: str):
        if mode == 'norm':
            self._get_statistics(x)
            x = self._normalize(x)

        elif mode == 'denorm':
            x = self._denormalize(x)

        else:
            raise NotImplementedError

        return x

    def _init_params(self):
        # initialize RevIN params: (C,)
        self.affine_weight = nn.Parameter(torch.ones(self.num_features))
        self.affine_bias = nn.Parameter(torch.zeros(self.num_features))

    def _get_statistics(self, x):
        dim2reduce = tuple(range(1, x.ndim - 1))
        self.mean = torch.mean(x, dim=dim2reduce, keepdim=True).detach()
        self.stdev = torch.sqrt(torch.var(x, dim=dim2reduce, keepdim=True, unbiased=False) + self.eps).detach()

    def _normalize(self, x):
        x = x - self.mean
        x = x / self.stdev
        if self.affine:
            x = x * self.affine_weight
            x = x + self.affine_bias

        return x

    def _denormalize(self, x):
        if self.affine:
            x = x - self.affine_bias
            x = x / (self.affine_weight + self.eps * self.eps)
        x = x * self.stdev
        x = x + self.mean

        return x

# 模型的类，在这里加n
class BasicLinear(nn.Module):

    def __init__(self, hist_len, pred_len, var_num, rev=True, drop=0.1, individual=False, layer_type='linear', 
                 n_grid=5, degree=None, order=None, n_center=None,Bn=3):
    # layer_type='linear'是层的类型
        super(BasicLinear, self).__init__()
        print("the whole pred_len",pred_len)

    # 第一次出现KAN的位置
        
        if individual:
            self.model = nn.ModuleList(
                [KANInterface(hist_len, pred_len, layer_type, n_grid, degree, order, n_center, Bn=Bn) for _ in range(var_num)]) # 多层KAN，var_num是层数
        else:
            self.model = KANInterface(hist_len, pred_len, layer_type, n_grid, degree, order, n_center, Bn=Bn)
        self.dropout = nn.Dropout(drop)
        self.rev = RevIN(var_num) if rev else None
        self.individual = individual
        #print("k in rlinear", k)


    def forward(self, var_x, marker_x):
        var_x = var_x[..., 0]  # x: [B, L, D]
        var_x = self.rev(var_x, 'norm') if self.rev else var_x
        var_x = self.dropout(var_x)
        if self.individual:
            pred = []
            for idx, proj in enumerate(self.model):
                pred.append(proj(var_x[:, :, idx]))
            pred = torch.stack(pred, dim=-1)
        else:
            pred = self.model(var_x.transpose(1, 2)).transpose(1, 2)
        pred = self.rev(pred, 'denorm') if self.rev else pred
        return pred
