import pandas as pd
from filelock import FileLock

def process_csv_dataset(args):
    """
    Process the CSV dataset if provided and update content_description.
    
    Args:
        args: Command line arguments
    
    Returns:
        Updated args with content_description set from CSV if applicable, and the DataFrame
    """
    if not args.dataset:
        return args, None
    
    try:
        # Validate row_index
        if args.row_index is None or args.row_index < 1 or args.row_index > 50:
            raise ValueError("Row index must be between 1 and 50")
        
        # Read the CSV file
        df = pd.read_csv(args.dataset)
        
        # Check if the CSV has at least row_index rows
        if len(df) < args.row_index:
            raise ValueError(f"CSV file has only {len(df)} rows, but row_index is {args.row_index}")
        
        # Get the row (subtract 1 because row_index is 1-based, but DataFrame indices are 0-based)
        row = df.iloc[args.row_index - 1]
        
        # Get the first two columns
        if len(df.columns) < 2:
            raise ValueError("CSV file must have at least 2 columns")
        
        col1_name = df.columns[0]
        col2_name = df.columns[1]
        
        # Set content_description from the CSV data
        args.content_description = f"{row[col1_name]} - {row[col2_name]}"
        print(f"Using content description from CSV: '{args.content_description}'")
        
        return args, df
    
    except Exception as e:
        print(f"Error processing CSV dataset: {e}")
        raise

def update_csv_with_results(csv_path, row_index, scores, folder_name):
    """
    Update the CSV file with scores.
    
    Args:
        csv_path: Path to the CSV file
        row_index: Row index to update (1-based)
        scores: Dictionary of scores
        folder_name: Current folder name
    """
    try:
        with FileLock(csv_path + ".lock"):
            # Read the CSV file
            df = pd.read_csv(csv_path)
            
            # Check if the row exists
            if len(df) < row_index:
                raise ValueError(f"CSV file has only {len(df)} rows, but row_index is {row_index}")
            
            # Add columns if they don't exist
            for score_name, score_value in scores.items():
                column_name = f"{folder_name}_{score_name}"
                if column_name not in df.columns:
                    df[column_name] = None
                
                # Check if the cell is already filled
                if pd.notna(df.at[row_index - 1, column_name]):
                    print(f"Warning: Column '{column_name}' at row {row_index} already has a value. Overwriting.")
                
                # Update the score
                df.at[row_index - 1, column_name] = score_value
            
            # Save the updated CSV
            df.to_csv(csv_path, index=False)
            print(f"Successfully updated CSV file with scores at row {row_index}")
            
    except Exception as e:
        print(f"Error updating CSV file: {e}")
        raise

def update_csv_with_relative_results(csv_path, row_index, comparison_results, comparison_prefix):
    """
    Update the CSV file with relative comparison results.
    
    Args:
        csv_path: Path to the CSV file
        row_index: Row index to update (1-based)
        comparison_results: Dictionary of comparison results
        comparison_prefix: 
    """
    try:
        with FileLock(csv_path + ".lock"):
            # Read the CSV file
            df = pd.read_csv(csv_path)
            
            # Check if the row exists
            if len(df) < row_index:
                raise ValueError(f"CSV file has only {len(df)} rows, but row_index is {row_index}")
            
            # Add columns if they don't exist
            for result_name, result_value in comparison_results.items():
                column_name = f"{comparison_prefix}_{result_name}"
                if column_name not in df.columns:
                    df[column_name] = None
                
                # Check if the cell is already filled
                if pd.notna(df.at[row_index - 1, column_name]):
                    print(f"Warning: Column '{column_name}' at row {row_index} already has a value. Overwriting.")
                
                # Update the result
                df.at[row_index - 1, column_name] = result_value
            
            # Save the updated CSV
            df.to_csv(csv_path, index=False)
            print(f"Successfully updated CSV file with comparison results at row {row_index}")
        
    except Exception as e:
        print(f"Error updating CSV file with comparison results: {e}")
        raise
