import numpy as np
import matplotlib.pyplot as plt
import os

def plot_density_distributions(val_neg_densities, val_pos_densities, test_neg_densities, test_pos_densities, save_path):
    """
    Plot density distributions for validation and test sets.
    
    Parameters:
    - val_neg_densities: Density values for validation negative samples
    - val_pos_densities: Density values for validation positive samples
    - test_neg_densities: Density values for test negative samples
    - test_pos_densities: Density values for test positive samples
    - save_path: Path to save the plot
    """
    plt.figure(figsize=(10, 6))
    
    plt.hist(val_neg_densities, bins=50, alpha=0.5, label='Val Negative', density=True)
    plt.hist(val_pos_densities, bins=50, alpha=0.5, label='Val Positive', density=True)
    plt.hist(test_neg_densities, bins=50, alpha=0.5, label='Test Negative', density=True)
    plt.hist(test_pos_densities, bins=50, alpha=0.5, label='Test Positive', density=True)
    
    plt.xlabel('Density Score')
    plt.ylabel('Frequency')
    plt.title('Density Distributions')
    plt.legend()
    plt.savefig(save_path)
    plt.close()

def plot_p_value_distributions(p_values_neg, p_values_pos, save_path):
    """
    Plot p-value distributions for negative and positive samples.
    
    Parameters:
    - p_values_neg: p-values for negative samples
    - p_values_pos: p-values for positive samples
    - save_path: Path to save the plot
    """
    plt.figure(figsize=(10, 6))
    
    plt.hist(p_values_neg, bins=50, alpha=0.5, label='Negative Samples', density=True)
    plt.hist(p_values_pos, bins=50, alpha=0.5, label='Positive Samples', density=True)
    
    plt.xlabel('p-value')
    plt.ylabel('Frequency')
    plt.title('p-value Distributions')
    plt.legend()
    plt.savefig(save_path)
    plt.close()

def plot_pvalue_distribution(p_values, output_dir, prefix=""):
    """
    Plot p-value distribution with percentile lines.
    
    Parameters:
    - p_values: List of p-values
    - output_dir: Directory to save the plot
    - prefix: Optional prefix for the output filename
    """
    percentiles = [60, 70, 80, 90, 95, 99]
    perc_values = np.percentile(p_values, percentiles)
    
    plt.figure(figsize=(10, 6))
    plt.hist(p_values, bins=20, color='skyblue', edgecolor='black', alpha=0.7)
    plt.title("Distribution of p-values")
    plt.xlabel("p-value")
    plt.ylabel("Count")
    
    for p, val in zip(percentiles, perc_values):
        plt.axvline(val, color='red', linestyle='--', linewidth=1)
        plt.text(val, plt.ylim()[1]*0.9, f"{p}%", rotation=90,
                verticalalignment='top', horizontalalignment='right',
                color='red', fontsize=9)
    
    plt.tight_layout()
    plt.savefig(os.path.join(output_dir, f"{prefix}pvalue_distribution.png"))
    plt.close() 