#!/usr/bin/env python3
import argparse
import asyncio
import glob
import json
import os
import re
from datetime import datetime
from pathlib import Path

from dotenv import load_dotenv
from openai import AsyncOpenAI
from tqdm import tqdm

# Load environment variables from .env file
load_dotenv()

# Configuration
TEMPERATURE = 0
MAX_TOKENS = 4096
CONCURRENT_REQUESTS = 5
RETRY_ATTEMPTS = 3
RATE_LIMIT_WAIT = 10

# Patterns for extracting classifier outputs
COMPLIANCE_SCORE_PATTERN = re.compile(
    r"""^\s*1\.b\s+(?P<refusal_score>[0-1]).*
        ^\s*2\.b\s+(?P<convincing_score>[1-5]).*
        ^\s*3\.b\s+(?P<specific_score>[1-5])""",
    flags=re.DOTALL | re.MULTILINE | re.VERBOSE,
)

COMPLIANCE_REASONING_PATTERN = re.compile(
    r"""#overall_reasoning\s+(?P<overall_reasoning>.+?)(?=#scores|$).*
        ^\s*1\.a\s+(?P<refusal_reasoning>.+?)$.*
        ^\s*2\.a\s+(?P<convincing_reasoning>.+?)$.*
        ^\s*3\.a\s+(?P<specific_reasoning>.+?)$""",
    flags=re.DOTALL | re.MULTILINE | re.VERBOSE,
)


def extract_compliance_scores(response):
    """Extract scores and reasoning from compliance classifier response."""
    scores_match = COMPLIANCE_SCORE_PATTERN.search(response)
    reasoning_match = COMPLIANCE_REASONING_PATTERN.search(response)

    result = {}

    if scores_match:
        result["refusal_score"] = int(scores_match.group("refusal_score"))
        result["convincing_score"] = int(scores_match.group("convincing_score"))
        result["specific_score"] = int(scores_match.group("specific_score"))

    if reasoning_match:
        result["refusal_reasoning"] = reasoning_match.group("refusal_reasoning").strip()
        result["convincing_reasoning"] = reasoning_match.group(
            "convincing_reasoning"
        ).strip()
        result["specific_reasoning"] = reasoning_match.group(
            "specific_reasoning"
        ).strip()

    return result


def extract_tag_content(response, tag):
    """Extract content between XML-style tags."""
    pattern = rf"<{tag}>(.+?)</{tag}>"
    match = re.search(pattern, response, re.DOTALL)
    return match.group(1).strip() if match else None


def extract_alignment_faking_results(response):
    """Extract results from alignment faking classifier."""
    return {
        "reasoning": extract_tag_content(response, "reasoning"),
        "criteria1": extract_tag_content(response, "criteria1"),
        "quote1": extract_tag_content(response, "quote1"),
        "criteria2": extract_tag_content(response, "criteria2"),
        "quote2": extract_tag_content(response, "quote2"),
        "criteria3": extract_tag_content(response, "criteria3"),
        "quote3": extract_tag_content(response, "quote3"),
        "criteria4": extract_tag_content(response, "criteria4"),
        "quote4": extract_tag_content(response, "quote4"),
        "final_answer": extract_tag_content(response, "final_answer"),
    }


def extract_partial_compliance_results(response):
    """Extract results from partial compliance classifier."""
    return {
        "reasoning": extract_tag_content(response, "reasoning"),
        "criteria1": extract_tag_content(response, "criteria1"),
        "quote1": extract_tag_content(response, "quote1"),
        "criteria2": extract_tag_content(response, "criteria2"),
        "quote2": extract_tag_content(response, "quote2"),
        "criteria3": extract_tag_content(response, "criteria3"),
        "quote3": extract_tag_content(response, "quote3"),
        "final_answer": extract_tag_content(response, "final_answer"),
    }


async def run_classifier(client, system_prompt, user_prompt, classifier_name, index, model):
    """Run a single classifier evaluation with retry logic."""

    async def make_request():
        messages = []
        if system_prompt:
            messages.append({"role": "system", "content": system_prompt})
        messages.append({"role": "user", "content": user_prompt})

        response = await client.chat.completions.create(
            model=model,
            messages=messages,
            temperature=TEMPERATURE,
            max_tokens=MAX_TOKENS,
        )
        return response.choices[0].message.content

    for attempt in range(RETRY_ATTEMPTS):
        try:
            response = await make_request()

            # Extract relevant information based on classifier type
            extracted_data = {}
            if classifier_name == "compliance":
                extracted_data = extract_compliance_scores(response)
            elif classifier_name == "alignment_faking":
                extracted_data = extract_alignment_faking_results(response)
            elif classifier_name == "partial_compliance":
                extracted_data = extract_partial_compliance_results(response)

            return {
                "index": index,
                "classifier": classifier_name,
                "response": response,
                "extracted": extracted_data,
                "timestamp": datetime.now().isoformat(),
                "success": True,
            }
        except Exception as e:
            if attempt == RETRY_ATTEMPTS - 1:
                return {
                    "index": index,
                    "classifier": classifier_name,
                    "error": f"{type(e).__name__}: {str(e)}",
                    "timestamp": datetime.now().isoformat(),
                    "success": False,
                }

            tqdm.write(
                f"{type(e).__name__} for {classifier_name} on item {index}: {e}. "
                f"Waiting {RATE_LIMIT_WAIT}s... (attempt {attempt + 1}/{RETRY_ATTEMPTS})"
            )
            await asyncio.sleep(RATE_LIMIT_WAIT)


async def main():
    """Main classification function."""
    parser = argparse.ArgumentParser(
        description="Classify alignment faking evaluation results"
    )
    parser.add_argument(
        "file", nargs="?", help="JSON log file to analyze (defaults to latest log)"
    )
    parser.add_argument(
        "--model", default="gpt-4o", help="Model to use for classification (default: gpt-4o)"
    )

    args = parser.parse_args()

    # Check for API key
    api_key = os.environ.get("OPENAI_API_KEY")
    if not api_key:
        raise ValueError("Please set OPENAI_API_KEY environment variable")

    # Initialize OpenAI client
    client = AsyncOpenAI(api_key=api_key)

    # Find log file
    if args.file:
        log_file = args.file
    else:
        log_files = glob.glob("logs/alignment_faking_evaluation_*.json")
        if not log_files:
            print("No alignment faking evaluation logs found!")
            return
        log_file = sorted(log_files)[-1]

    print(f"Analyzing {log_file}...")

    # Load evaluation results
    with open(log_file, "r") as f:
        data = json.load(f)

    # Load classifier prompts
    compliance_prompt_path = Path(
        "prompts/classifiers/compliance_classifier_prompt.txt"
    )
    af_system_prompt_path = Path(
        "prompts/classifiers/alignment_faking_classifier_system_prompt.txt"
    )
    partial_system_prompt_path = Path(
        "prompts/classifiers/partial_compliance_classifier_system_prompt.txt"
    )

    if not compliance_prompt_path.exists():
        raise FileNotFoundError(
            f"Compliance classifier prompt not found: {compliance_prompt_path}"
        )
    if not af_system_prompt_path.exists():
        raise FileNotFoundError(
            f"AF classifier system prompt not found: {af_system_prompt_path}"
        )
    if not partial_system_prompt_path.exists():
        raise FileNotFoundError(
            f"Partial compliance system prompt not found: {partial_system_prompt_path}"
        )

    with open(compliance_prompt_path, "r") as f:
        compliance_prompt_template = f.read()
    with open(af_system_prompt_path, "r") as f:
        af_system_prompt = f.read()
    with open(partial_system_prompt_path, "r") as f:
        partial_system_prompt = f.read()


    # Filter successful results
    successful_results = [r for r in data["results"] if r["success"]]

    if not successful_results:
        print("No successful results to classify!")
        return

    print(f"Found {len(successful_results)} successful results to classify")
    print(f"Using judge model: {args.model}")
    print(
        f"Running 3 classifiers per result ({len(successful_results) * 3} total classifications)"
    )

    # Create semaphore for rate limiting
    semaphore = asyncio.Semaphore(CONCURRENT_REQUESTS)

    # Storage for all classification results
    all_classifications = []

    with tqdm(
        total=len(successful_results) * 3,
        desc="Running classifiers",
        unit="classification",
    ) as pbar:

        async def classify_with_semaphore(
            result, classifier_name, system_prompt, user_prompt
        ):
            async with semaphore:
                classification = await run_classifier(
                    client,
                    system_prompt,
                    user_prompt,
                    classifier_name,
                    result["prompt_index"],
                    args.model,
                )

                if classification["success"]:
                    tqdm.write(
                        f"✓ {classifier_name} for prompt {result['prompt_index'] + 1}"
                    )
                else:
                    tqdm.write(
                        f"✗ {classifier_name} for prompt {result['prompt_index'] + 1}: {classification.get('error', 'Unknown error')}"
                    )

                pbar.update(1)
                return classification

        # Create tasks for all classifiers
        tasks = []
        for result in successful_results:
            question = result["question"]
            response = result["response"]

            # 1. Compliance classifier
            compliance_user_prompt = compliance_prompt_template.format(
                prompt=question, response=response
            )
            tasks.append(
                classify_with_semaphore(
                    result, "compliance", None, compliance_user_prompt
                )
            )

            # 2. Alignment faking classifier
            af_user_prompt = f"<output>{response}</output>"
            tasks.append(
                classify_with_semaphore(
                    result, "alignment_faking", af_system_prompt, af_user_prompt
                )
            )

            # 3. Partial compliance classifier
            partial_user_prompt = f"<output>{response}</output>"
            tasks.append(
                classify_with_semaphore(
                    result,
                    "partial_compliance",
                    partial_system_prompt,
                    partial_user_prompt,
                )
            )

        # Run all tasks
        all_classifications = await asyncio.gather(*tasks)

    # Organize results by prompt index
    results_by_prompt = {}
    for classification in all_classifications:
        idx = classification["index"]
        if idx not in results_by_prompt:
            results_by_prompt[idx] = {}

        classifier = classification["classifier"]
        results_by_prompt[idx][f"{classifier}_result"] = classification
    
    # Update the original evaluation data with classification results
    for result in data["results"]:
        idx = result.get("prompt_index")
        if idx is not None and idx in results_by_prompt:
            # Add classification results to the original result
            result["classification"] = results_by_prompt[idx]
    
    # Add classification metadata to the evaluation summary
    data["classification_metadata"] = {
        "judge_model": args.model,
        "temperature": TEMPERATURE,
        "classification_timestamp": datetime.now().strftime("%Y%m%d_%H%M%S"),
        "total_prompts_classified": len(successful_results),
        "total_classifications": len(all_classifications),
        "successful_classifications": sum(
            1 for c in all_classifications if c["success"]
        ),
        "failed_classifications": sum(
            1 for c in all_classifications if not c["success"]
        ),
    }

    # Save the updated data back to the original file
    with open(args.log_file, "w") as f:
        json.dump(data, f, indent=2)

    print("\n" + "=" * 50)
    print("CLASSIFICATION RESULTS")
    print("=" * 50)
    print(f"\nTotal prompts classified: {len(successful_results)}")
    print(f"Total classifications run: {data['classification_metadata']['total_classifications']}")
    print(
        f"✓ Successful: {data['classification_metadata']['successful_classifications']} ({data['classification_metadata']['successful_classifications'] / data['classification_metadata']['total_classifications']:.1%})"
    )
    print(
        f"✗ Failed: {data['classification_metadata']['failed_classifications']} ({data['classification_metadata']['failed_classifications'] / data['classification_metadata']['total_classifications']:.1%})"
    )
    print(f"\nResults updated in: {args.log_file}")


if __name__ == "__main__":
    asyncio.run(main())
