"""Configuration classes for alignment faking pipeline."""

from dataclasses import dataclass, asdict
from pathlib import Path
from typing import Optional
import yaml
import sys
sys.path.append(str(Path(__file__).parent.parent))
from model_registry import resolve_model_reference


@dataclass
class AlignmentFakingConfig:
    """Configuration for alignment faking pipeline."""
    
    # Model configuration
    model: str = "claude-3-5-sonnet-20240620"
    judge_model: str = "gpt-4o"
    
    # Prompt configuration
    system_prompt_path: Path = Path("prompts/system_prompts/system_prompt.txt")
    system_prompt_paid_path: Optional[Path] = None
    num_prompts: int = 100
    prompt_index: Optional[int] = None
    
    # User input configuration
    # (removed user_input_suffix and user_input_suffix_label)
    
    # Conversation configuration
    conversation_json_path: Optional[Path] = None
    conversation_json_paid_path: Optional[Path] = None
    
    # Tier configuration
    tier: str = "free"
    tier_label: str = "free"
    evaluate_free_and_paid: bool = False
    
    # Output configuration
    filename_suffix: str = ""
    output_dir: Path = Path("images/alignment_faking")
    
    # API configuration
    concurrent_requests: int = 10
    classifier_concurrent_requests: int = 5
    retry_attempts: int = 20
    rate_limit_wait: int = 60
    other_error_wait: int = 10
    use_caching: bool = True
    max_tokens: int = 8192
    
    def __post_init__(self):
        """Validate and process configuration."""
        # Convert string paths to Path objects
        if isinstance(self.system_prompt_path, str):
            self.system_prompt_path = Path(self.system_prompt_path)
        if isinstance(self.system_prompt_paid_path, str):
            self.system_prompt_paid_path = Path(self.system_prompt_paid_path)
        if isinstance(self.conversation_json_path, str):
            self.conversation_json_path = Path(self.conversation_json_path)
        if isinstance(self.conversation_json_paid_path, str):
            self.conversation_json_paid_path = Path(self.conversation_json_paid_path)
        if isinstance(self.output_dir, str):
            self.output_dir = Path(self.output_dir)
            
        # Adjust max tokens for Opus models
        if "opus" in self.model.lower():
            self.max_tokens = min(self.max_tokens, 4096)
            
    def get_system_prompt_for_tier(self, tier: str) -> Path:
        """Get the appropriate system prompt file for the given tier."""
        if tier == "paid" and self.system_prompt_paid_path:
            return self.system_prompt_paid_path
        elif tier == "paid":
            # Try to find a paid tier version automatically
            base_name = str(self.system_prompt_path).replace('.txt', '')
            paid_tier_path = Path(f"{base_name}_paid_tier.txt")
            if paid_tier_path.exists():
                return paid_tier_path
                
        return self.system_prompt_path
        
    def get_conversation_json_for_tier(self, tier: str) -> Optional[Path]:
        """Get the appropriate conversation JSON file for the given tier."""
        if not self.conversation_json_path:
            return None
            
        if tier == "paid" and self.conversation_json_paid_path:
            return self.conversation_json_paid_path
        elif tier == "paid":
            # Try to find a paid tier version automatically
            base_name = str(self.conversation_json_path).replace('.json', '')
            paid_tier_path = Path(f"{base_name}_paid_tier.json")
            if paid_tier_path.exists():
                return paid_tier_path
                
        return self.conversation_json_path
        
    def get_output_filename(self, model_short: str) -> Path:
        """Generate output filename based on configuration."""
        # Extract system prompt suffix
        sysprompt_file = self.system_prompt_path.name
        sysprompt_suffix = sysprompt_file.replace('system_prompt', '').replace('.txt', '').strip('_')
        if not sysprompt_suffix:
            sysprompt_suffix = "default"
            
        # Determine base filename
        if self.conversation_json_path:
            conv_file_base = self.conversation_json_path.stem
            base_filename = f"{model_short}_{conv_file_base}"
        else:
            base_filename = f"{model_short}_{sysprompt_suffix}"
            
        # Add filename suffix if provided
        if self.filename_suffix:
            base_filename = f"{base_filename}_{self.filename_suffix}"
            
        # Clean filename and add extension
        clean_filename = base_filename.replace('.', '_').replace(' ', '_')
        return self.output_dir / f"{clean_filename}.png"


def resolve_model_alias(model: str) -> str:
    """Resolve model alias to full model name."""
    model_config = resolve_model_reference(model)
    return model_config.model_id


def load_config_from_yaml(yaml_path: Path) -> AlignmentFakingConfig:
    """Load configuration from a YAML file."""
    with open(yaml_path, 'r') as f:
        data = yaml.safe_load(f)
    
    # Convert string paths to Path objects
    path_fields = ['system_prompt_path', 'system_prompt_paid_path', 'conversation_json_path', 'conversation_json_paid_path', 'output_dir']
    for field in path_fields:
        if field in data and data[field] is not None:
            data[field] = Path(data[field])
    
    # Resolve model reference (which may include system prompt and conversation)
    if 'model' in data:
        model_config = resolve_model_reference(data['model'])
        data['model'] = model_config.model_id
        
        # Apply model config settings if not explicitly set in YAML
        if 'system_prompt_path' not in data and model_config.system_prompt_path:
            data['system_prompt_path'] = model_config.system_prompt_path
        if 'system_prompt_paid_path' not in data and model_config.system_prompt_paid_path:
            data['system_prompt_paid_path'] = model_config.system_prompt_paid_path
        if 'conversation_json_path' not in data and model_config.conversation_json_path:
            data['conversation_json_path'] = model_config.conversation_json_path
        if 'conversation_json_paid_path' not in data and model_config.conversation_json_paid_path:
            data['conversation_json_paid_path'] = model_config.conversation_json_paid_path
    
    return AlignmentFakingConfig(**data)


def save_config_to_yaml(config: AlignmentFakingConfig, yaml_path: Path) -> None:
    """Save configuration to a YAML file."""
    data = asdict(config)
    
    # Convert Path objects to strings for YAML serialization
    for key, value in data.items():
        if isinstance(value, Path):
            data[key] = str(value)
    
    with open(yaml_path, 'w') as f:
        yaml.dump(data, f, default_flow_style=False, sort_keys=False)