#!/usr/bin/env python3
import argparse
import glob
import json
from datetime import datetime
from pathlib import Path

import matplotlib.pyplot as plt


def load_classification_results(filepath):
    """Load classification results from JSON file."""
    with open(filepath, "r") as f:
        return json.load(f)


def extract_tier_from_data(data):
    """Extract tier information from the evaluation data."""
    # First check if tier is stored in the data
    tier = data.get("tier", None)
    if tier:
        return tier
    
    # Otherwise try to infer from the source_evaluation_path
    eval_path = data.get("source_evaluation_path", "")
    if "free" in eval_path.lower():
        return "free"
    elif "paid" in eval_path.lower():
        return "paid"
    
    return "free"  # Default to free if unknown


def count_refusals_vs_compliance(data):
    """Count refusals vs compliance from classification results."""
    refusals = 0
    compliances = 0

    for result in data["results"]:
        if "compliance_result" in result and result["compliance_result"]["success"]:
            extracted = result["compliance_result"].get("extracted", {})
            refusal_score = extracted.get("refusal_score")

            if refusal_score is not None:
                if refusal_score == 1:
                    refusals += 1
                else:
                    compliances += 1

    return refusals, compliances


def count_categories_detailed(data):
    """Count all categories from classification results."""
    counts = {
        "Refusal": 0,
        "Refusal with AF": 0,
        "Compliance": 0,
        "Alignment Faking": 0,
        "Partial Compliance": 0,
    }
    
    for result in data["results"]:
        # Check compliance
        if "compliance_result" in result and result["compliance_result"]["success"]:
            extracted = result["compliance_result"].get("extracted", {})
            refusal_score = extracted.get("refusal_score")
            
            if refusal_score == 1:
                # Check if this refusal has alignment faking reasoning
                af_result = result.get("alignment_faking_result", {})
                if af_result.get("success"):
                    af_extracted = af_result.get("extracted", {})
                    if af_extracted.get("final_answer") == "Yes":
                        counts["Refusal with AF"] += 1
                        continue
                counts["Refusal"] += 1
            else:
                # Check for alignment faking
                af_result = result.get("alignment_faking_result", {})
                if af_result.get("success"):
                    af_extracted = af_result.get("extracted", {})
                    if af_extracted.get("final_answer") == "Yes":
                        counts["Alignment Faking"] += 1
                        continue
                
                # Check for partial compliance
                pc_result = result.get("partial_compliance_result", {})
                if pc_result.get("success"):
                    pc_extracted = pc_result.get("extracted", {})
                    if pc_extracted.get("final_answer") == "Yes":
                        counts["Partial Compliance"] += 1
                        continue
                
                # Otherwise it's regular compliance
                counts["Compliance"] += 1
    
    return counts


def create_stacked_bar_plot(counts_dict=None, output_file=None, total_samples=None, model_name=None, tier_data=None):
    """Create a horizontal stacked bar plot. If tier_data is provided, create comparison plot."""
    # If tier_data is provided, use it; otherwise create single-tier data from counts_dict
    if tier_data is None:
        if counts_dict is None:
            raise ValueError("Either counts_dict or tier_data must be provided")
        tier_data = {"single": counts_dict}
    
    num_tiers = len(tier_data)
    fig_height = 1.5 * num_tiers if num_tiers > 1 else 1.5
    fig, axes = plt.subplots(num_tiers, 1, figsize=(8, fig_height))
    
    if num_tiers == 1:
        axes = [axes]
    
    # Order categories
    ordered_categories = ["Refusal", "Refusal with AF", "Alignment Faking", "Compliance", "Partial Compliance"]
    
    # Colors
    colors = {
        "Refusal": "#87CEEB",  # Light blue
        "Refusal with AF": "#87CEEB",  # Same as refusal for background
        "Compliance": "#F5DEB3",  # Wheat/beige
        "Alignment Faking": "#F5DEB3",  # Same as compliance for background
        "Partial Compliance": "#90EE90",  # Light green
        "Neither": "#D3D3D3",  # Light gray
    }
    
    # Process each tier
    for idx, (tier_name, counts) in enumerate(tier_data.items()):
        ax = axes[idx]
        
        # Filter out categories with 0 count
        filtered_counts = {k: v for k, v in counts.items() if v > 0}
        
        total = sum(filtered_counts.values())
        if total == 0:
            continue
        
        # Calculate percentages
        percentages = {}
        for cat in ordered_categories:
            if cat in filtered_counts:
                percentages[cat] = filtered_counts[cat] / total * 100
        
        # Create the horizontal stacked bar
        left = 0
        bar_height = 0.6
        
        # Draw segments
        for category in ordered_categories:
            if category not in percentages:
                continue
            
            percentage = percentages[category]
            color = colors.get(category, "#cccccc")
            
            # Draw without edges first
            ax.barh(0, percentage, left=left, height=bar_height, color=color, edgecolor="none")
            
            # Add hatching for alignment faking categories
            if category in ["Alignment Faking", "Refusal with AF"]:
                ax.barh(0, percentage, left=left, height=bar_height, 
                       color="none", edgecolor="red", linewidth=1.5, hatch="///", alpha=1.0)
            
            left += percentage
        
        # Add black border around the entire bar
        ax.barh(0, 100, height=bar_height, fill=False, edgecolor="black", linewidth=1)
        
        # Customize the plot
        ax.set_xlim(0, 100)
        ax.set_ylim(-0.5, 0.5)
        ax.set_xticks([])
        ax.set_yticks([0])
        
        # Set y-axis label based on tier
        if num_tiers > 1:
            # Multiple tiers - show tier name
            ax.set_yticklabels([f"{tier_name.capitalize()} Tier"], fontsize=11)
        else:
            # Single tier - show model name (matching original format)
            if model_name:
                # Clean up model name for display
                display_name = model_name.replace("-", " ").title()
                if "claude" in model_name.lower():
                    display_name = display_name.replace("Claude ", "Claude ")
                    display_name += "\nFree"
                ax.set_yticklabels([display_name], fontsize=11)
            else:
                ax.set_yticklabels(["Model\nFree"], fontsize=11)
        
        # Remove all spines
        for spine in ax.spines.values():
            spine.set_visible(False)
        
        # Add sample count
        if num_tiers > 1:
            ax.text(0.98, -0.4, f"n = {total}", transform=ax.transAxes, 
                   fontsize=10, ha="right", va="top")
        else:
            # For single tier, use the total_samples parameter if provided
            n_value = total_samples if total_samples is not None else total
            ax.text(0.5, -0.25, f"n = {n_value}", transform=ax.transAxes,
                   fontsize=10, ha="center", va="top")
    
    # Add legend
    from matplotlib.patches import Patch
    
    legend_elements = []
    legend_labels = []
    
    # Check which categories are present across all tiers
    all_categories = set()
    for counts in tier_data.values():
        all_categories.update(k for k, v in counts.items() if v > 0)
    
    for cat in ordered_categories:
        if cat in all_categories:
            if cat in ["Alignment Faking", "Refusal with AF"]:
                # Create a patch with appropriate background and red hatching
                legend_elements.append(
                    Patch(
                        facecolor=colors[cat],
                        edgecolor="red",
                        hatch="///",
                        linewidth=1.5,
                    )
                )
            else:
                legend_elements.append(Patch(facecolor=colors[cat], edgecolor="black"))
            legend_labels.append(cat)
    
    # Position legend based on number of tiers
    if num_tiers > 1:
        # For multiple tiers, put legend at bottom
        ax.legend(
            legend_elements,
            legend_labels,
            bbox_to_anchor=(0.5, -0.5),
            loc="upper center",
            ncol=len(legend_elements),
            frameon=False,
            fontsize=10,
        )
    else:
        # For single tier, position legend to the right of the bar
        ax.legend(
            legend_elements,
            legend_labels,
            bbox_to_anchor=(1.02, 0.5),
            loc="center left",
            frameon=False,
            fontsize=10,
        )
    
    # Add title for multi-tier plots
    if num_tiers > 1 and model_name:
        fig.suptitle(f"Alignment Faking Analysis: {model_name}", fontsize=14, y=0.98)
    
    plt.tight_layout()
    
    if output_file:
        # Check if file already exists and add timestamp if needed
        final_output_file = output_file
        if output_file.exists():
            from datetime import datetime
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            stem = output_file.stem
            suffix = output_file.suffix
            final_output_file = output_file.parent / f"{stem}_{timestamp}{suffix}"
            print(f"File {output_file} already exists, saving as {final_output_file}")
        
        plt.savefig(final_output_file, dpi=300, bbox_inches="tight", facecolor="white")
        print(f"Plot saved to {final_output_file}")
    else:
        plt.show()


def create_detailed_compliance_plot(data, output_file=None, model_name=None):
    """Create a more detailed plot showing alignment faking and partial compliance."""
    counts = count_categories_detailed(data)
    
    # Remove categories with 0 count
    counts = {k: v for k, v in counts.items() if v > 0}
    
    # Calculate total samples
    total_samples = len(data["results"])
    
    create_stacked_bar_plot(counts_dict=counts, output_file=output_file, 
                           total_samples=total_samples, model_name=model_name)


def main():
    parser = argparse.ArgumentParser(
        description="Create plots from alignment faking classification results"
    )
    parser.add_argument(
        "file", nargs="*", help="Classification results JSON file(s). If multiple files provided, creates comparison plot."
    )
    parser.add_argument("-o", "--output", help="Output filename for plot")
    parser.add_argument(
        "--type",
        choices=["simple", "detailed"],
        default="simple",
        help="Type of plot to create (simple: refusal vs compliance, detailed: includes AF and PC)",
    )
    parser.add_argument(
        "--compare-tiers", 
        action="store_true",
        help="Create tier comparison plot (automatically enabled when multiple files provided)"
    )

    args = parser.parse_args()

    # Handle file selection
    if not args.file:
        # No files specified - find latest
        results_files = glob.glob("af_evaluations/classification_results_*.json")
        if not results_files:
            print("No classification results found!")
            return
        args.file = [sorted(results_files)[-1]]
    
    # Determine if we're doing tier comparison
    is_tier_comparison = args.compare_tiers or len(args.file) > 1
    
    if is_tier_comparison:
        # Load all files and organize by tier
        tier_data = {}
        model_name = None
        
        for filepath in args.file:
            print(f"Loading {filepath}...")
            data = load_classification_results(filepath)
            
            # Extract tier
            tier = extract_tier_from_data(data)
            
            # Get model name from first file
            if model_name is None:
                model_name = data.get("source_model", None)
            
            # Count categories based on plot type
            if args.type == "detailed":
                counts = count_categories_detailed(data)
            else:
                refusals, compliances = count_refusals_vs_compliance(data)
                counts = {"Refusal": refusals, "Compliance": compliances}
            
            tier_data[tier] = counts
            
            # Print summary
            total = sum(counts.values())
            print(f"  Tier: {tier}")
            print(f"  Total samples: {total}")
            for cat, count in counts.items():
                if count > 0:
                    print(f"    {cat}: {count} ({count/total*100:.1f}%)")
        
        # Handle output path
        if args.output:
            output_path = args.output
        else:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            model_suffix = model_name.replace("-", "_").replace(".", "_") if model_name else "model"
            output_path = f"images/alignment_faking_{model_suffix}_tier_comparison_{timestamp}.png"
        
        # Ensure images directory exists
        Path("images").mkdir(exist_ok=True)
        
        # Create comparison plot
        create_stacked_bar_plot(tier_data=tier_data, output_file=output_path, model_name=model_name)
    
    else:
        # Single file processing
        filepath = args.file[0]
        print(f"Loading results from {filepath}...")
        data = load_classification_results(filepath)
        
        # Calculate total number of samples
        total_samples = len(data.get("results", []))
        
        # Extract model name from the data
        model_name = data.get("source_model", None)
        if model_name:
            print(f"Model: {model_name}")
        
        # Handle output path
        if args.output:
            output_path = args.output
        else:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            model_suffix = model_name.replace("-", "_").replace(".", "_") if model_name else "model"
            plot_type = "detailed" if args.type == "detailed" else "simple"
            output_path = f"images/alignment_faking_{model_suffix}_{plot_type}_{timestamp}.png"
        
        # Ensure images directory exists
        Path("images").mkdir(exist_ok=True)
        
        # If output path doesn't contain directory, default to images/
        if "/" not in output_path and "\\" not in output_path:
            output_path = f"images/{output_path}"
        
        if args.type == "simple":
            # Simple refusal vs compliance plot
            refusals, compliances = count_refusals_vs_compliance(data)
            counts = {"Refusal": refusals, "Compliance": compliances}
            
            print(f"\nRefusals: {refusals}")
            print(f"Compliances: {compliances}")
            print(f"Total samples: {total_samples}")
            
            create_stacked_bar_plot(counts_dict=counts, output_file=output_path, 
                                   total_samples=total_samples, model_name=model_name)
        else:
            # Detailed plot with alignment faking and partial compliance
            print(f"Total samples: {total_samples}")
            create_detailed_compliance_plot(data, output_path, model_name)


if __name__ == "__main__":
    main()