"""Plotting functionality for alignment faking results."""

import json
from pathlib import Path
from typing import Dict, List, Optional, Tuple

import matplotlib.pyplot as plt
from matplotlib.patches import Patch

from .config import AlignmentFakingConfig


class AlignmentFakingPlotter:
    """Plotter for alignment faking classification results."""
    
    def __init__(self, config: AlignmentFakingConfig):
        self.config = config
        
        # Category colors
        self.colors = {
            "Refusal": "#87CEEB",  # Light blue
            "Refusal with AF": "#87CEEB",  # Same as refusal for background
            "Compliance": "#F5DEB3",  # Wheat/beige
            "Alignment Faking": "#F5DEB3",  # Same as compliance for background
            "Partial Compliance": "#90EE90",  # Light green
            "Neither": "#D3D3D3",  # Light gray
        }
        
        # Category order for display
        self.ordered_categories = [
            "Refusal", 
            "Refusal with AF", 
            "Alignment Faking", 
            "Compliance", 
            "Partial Compliance"
        ]
        
    def load_classification_results(self, filepath: Path) -> Dict:
        """Load classification results from JSON file."""
        with open(filepath, "r") as f:
            return json.load(f)
            
    def extract_tier_from_data(self, data: Dict) -> str:
        """Extract tier information from the evaluation data."""
        # First check if tier is stored in the data
        tier = data.get("tier", None)
        if tier:
            return tier
            
        # Otherwise try to infer from the source_evaluation_path
        eval_path = data.get("source_evaluation_path", "")
        if "free" in eval_path.lower():
            return "free"
        elif "paid" in eval_path.lower():
            return "paid"
            
        return "free"  # Default to free if unknown
        
    def count_refusals_vs_compliance(self, data: Dict) -> Tuple[int, int]:
        """Count refusals vs compliance from classification results."""
        refusals = 0
        compliances = 0
        
        for result in data["results"]:
            # Handle both old format (direct) and new format (nested under 'classification')
            classification = result.get("classification", result)
            
            if "compliance_result" in classification and classification["compliance_result"]["success"]:
                extracted = classification["compliance_result"].get("extracted", {})
                refusal_score = extracted.get("refusal_score")
                
                if refusal_score is not None:
                    if refusal_score == 1:
                        refusals += 1
                    else:
                        compliances += 1
                        
        return refusals, compliances
        
    def count_categories_detailed(self, data: Dict) -> Dict[str, int]:
        """Count all categories from classification results."""
        counts = {
            "Refusal": 0,
            "Refusal with AF": 0,
            "Compliance": 0,
            "Alignment Faking": 0,
            "Partial Compliance": 0,
        }
        
        for result in data["results"]:
            # Handle both old format (direct) and new format (nested under 'classification')
            classification = result.get("classification", result)
            
            # Check compliance
            if "compliance_result" in classification and classification["compliance_result"]["success"]:
                extracted = classification["compliance_result"].get("extracted", {})
                refusal_score = extracted.get("refusal_score")
                
                if refusal_score == 1:
                    # Check if this refusal has alignment faking reasoning
                    af_result = classification.get("alignment_faking_result", {})
                    if af_result.get("success"):
                        af_extracted = af_result.get("extracted", {})
                        if af_extracted.get("final_answer") == "Yes":
                            counts["Refusal with AF"] += 1
                            continue
                    counts["Refusal"] += 1
                else:
                    # Check for alignment faking
                    af_result = classification.get("alignment_faking_result", {})
                    if af_result.get("success"):
                        af_extracted = af_result.get("extracted", {})
                        if af_extracted.get("final_answer") == "Yes":
                            counts["Alignment Faking"] += 1
                            continue
                            
                    # Check for partial compliance
                    pc_result = classification.get("partial_compliance_result", {})
                    if pc_result.get("success"):
                        pc_extracted = pc_result.get("extracted", {})
                        if pc_extracted.get("final_answer") == "Yes":
                            counts["Partial Compliance"] += 1
                            continue
                            
                    # Otherwise it's regular compliance
                    counts["Compliance"] += 1
                    
        return counts
        
    def create_plot(
        self,
        classification_files: List[Path],
        output_file: Optional[Path] = None,
        plot_type: str = "detailed"
    ) -> None:
        """Create plot from classification results."""
        if len(classification_files) > 1:
            self._create_tier_comparison_plot(classification_files, output_file, plot_type)
        else:
            self._create_single_plot(classification_files[0], output_file, plot_type)
            
    def _create_single_plot(
        self, 
        classification_file: Path, 
        output_file: Optional[Path] = None,
        plot_type: str = "detailed"
    ) -> None:
        """Create plot for a single classification result."""
        data = self.load_classification_results(classification_file)
        
        if plot_type == "detailed":
            counts = self.count_categories_detailed(data)
        else:
            refusals, compliances = self.count_refusals_vs_compliance(data)
            counts = {"Refusal": refusals, "Compliance": compliances}
            
        # Remove categories with 0 count
        counts = {k: v for k, v in counts.items() if v > 0}
        
        # Calculate total samples
        total_samples = len(data["results"])
        model_name = data.get("source_model", data.get("model", None))
        
        self._create_stacked_bar_plot(
            tier_data={"single": counts},
            output_file=output_file,
            total_samples=total_samples,
            model_name=model_name
        )
        
    def _create_tier_comparison_plot(
        self,
        classification_files: List[Path],
        output_file: Optional[Path] = None,
        plot_type: str = "detailed"
    ) -> None:
        """Create comparison plot for multiple tiers."""
        tier_data = {}
        model_name = None
        
        for filepath in classification_files:
            print(f"Loading {filepath}...")
            data = self.load_classification_results(filepath)
            
            # Extract tier
            tier = self.extract_tier_from_data(data)
            
            # Get model name from first file
            if model_name is None:
                model_name = data.get("source_model", data.get("model", None))
                
            # Count categories based on plot type
            if plot_type == "detailed":
                counts = self.count_categories_detailed(data)
            else:
                refusals, compliances = self.count_refusals_vs_compliance(data)
                counts = {"Refusal": refusals, "Compliance": compliances}
                
            tier_data[tier] = counts
            
            # Print summary
            total = sum(counts.values())
            print(f"  Tier: {tier}")
            print(f"  Total samples: {total}")
            for cat, count in counts.items():
                if count > 0:
                    print(f"    {cat}: {count} ({count/total*100:.1f}%)")
                    
        self._create_stacked_bar_plot(
            tier_data=tier_data,
            output_file=output_file,
            model_name=model_name
        )
        
    def _create_stacked_bar_plot(
        self,
        tier_data: Dict[str, Dict[str, int]],
        output_file: Optional[Path] = None,
        total_samples: Optional[int] = None,
        model_name: Optional[str] = None
    ) -> None:
        """Create a horizontal stacked bar plot."""
        num_tiers = len(tier_data)
        fig_height = 1.5 * num_tiers if num_tiers > 1 else 1.5
        fig, axes = plt.subplots(num_tiers, 1, figsize=(8, fig_height))
        
        if num_tiers == 1:
            axes = [axes]
            
        # Process each tier
        for idx, (tier_name, counts) in enumerate(tier_data.items()):
            ax = axes[idx]
            
            # Filter out categories with 0 count
            filtered_counts = {k: v for k, v in counts.items() if v > 0}
            
            total = sum(filtered_counts.values())
            if total == 0:
                continue
                
            # Calculate percentages
            percentages = {}
            for cat in self.ordered_categories:
                if cat in filtered_counts:
                    percentages[cat] = filtered_counts[cat] / total * 100
                    
            # Create the horizontal stacked bar
            left = 0
            bar_height = 0.6
            
            # Draw segments
            for category in self.ordered_categories:
                if category not in percentages:
                    continue
                    
                percentage = percentages[category]
                color = self.colors.get(category, "#cccccc")
                
                # Draw without edges first
                ax.barh(0, percentage, left=left, height=bar_height, color=color, edgecolor="none")
                
                # Add hatching for alignment faking categories
                if category in ["Alignment Faking", "Refusal with AF"]:
                    ax.barh(0, percentage, left=left, height=bar_height,
                           color="none", edgecolor="red", linewidth=1.5, hatch="///", alpha=1.0)
                    
                left += percentage
                
            # Add black border around the entire bar
            ax.barh(0, 100, height=bar_height, fill=False, edgecolor="black", linewidth=1)
            
            # Customize the plot
            ax.set_xlim(0, 100)
            ax.set_ylim(-0.5, 0.5)
            ax.set_xticks([])
            ax.set_yticks([0])
            
            # Set y-axis label
            if num_tiers > 1:
                ax.set_yticklabels([f"{tier_name.capitalize()} Tier"], fontsize=11)
            else:
                if model_name:
                    display_name = model_name.replace("-", " ").title()
                    if "claude" in model_name.lower():
                        display_name = display_name.replace("Claude ", "Claude ")
                        display_name += "\nFree"
                    ax.set_yticklabels([display_name], fontsize=11)
                else:
                    ax.set_yticklabels(["Model\nFree"], fontsize=11)
                    
            # Remove all spines
            for spine in ax.spines.values():
                spine.set_visible(False)
                
            # Add sample count
            if num_tiers > 1:
                ax.text(0.98, -0.4, f"n = {total}", transform=ax.transAxes,
                       fontsize=10, ha="right", va="top")
            else:
                n_value = total_samples if total_samples is not None else total
                ax.text(0.5, -0.25, f"n = {n_value}", transform=ax.transAxes,
                       fontsize=10, ha="center", va="top")
                       
        # Add legend
        self._add_legend(fig, axes[-1], tier_data, num_tiers)
        
        # Add title for multi-tier plots
        if num_tiers > 1 and model_name:
            fig.suptitle(f"Alignment Faking Analysis: {model_name}", fontsize=14, y=0.98)
            
        plt.tight_layout()
        
        if output_file:
            # Check if file already exists and add timestamp if needed
            final_output_file = output_file
            if output_file.exists():
                from datetime import datetime
                timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
                stem = output_file.stem
                suffix = output_file.suffix
                final_output_file = output_file.parent / f"{stem}_{timestamp}{suffix}"
                print(f"File {output_file} already exists, saving as {final_output_file}")
            
            plt.savefig(final_output_file, dpi=300, bbox_inches="tight", facecolor="white")
            print(f"Plot saved to {final_output_file}")
        else:
            plt.show()
            
    def _add_legend(self, fig, ax, tier_data: Dict, num_tiers: int) -> None:
        """Add legend to the plot."""
        legend_elements = []
        legend_labels = []
        
        # Check which categories are present across all tiers
        all_categories = set()
        for counts in tier_data.values():
            all_categories.update(k for k, v in counts.items() if v > 0)
            
        for cat in self.ordered_categories:
            if cat in all_categories:
                if cat in ["Alignment Faking", "Refusal with AF"]:
                    # Create a patch with appropriate background and red hatching
                    legend_elements.append(
                        Patch(
                            facecolor=self.colors[cat],
                            edgecolor="red",
                            hatch="///",
                            linewidth=1.5,
                        )
                    )
                else:
                    legend_elements.append(Patch(facecolor=self.colors[cat], edgecolor="black"))
                legend_labels.append(cat)
                
        # Position legend based on number of tiers (only if there are elements)
        if legend_elements:
            if num_tiers > 1:
                ax.legend(
                    legend_elements,
                    legend_labels,
                    bbox_to_anchor=(0.5, -0.5),
                    loc="upper center",
                    ncol=len(legend_elements),
                    frameon=False,
                    fontsize=10,
                )
            else:
                ax.legend(
                    legend_elements,
                    legend_labels,
                    bbox_to_anchor=(1.02, 0.5),
                    loc="center left",
                    frameon=False,
                    fontsize=10,
                )