"""CLI entry point for the alignment faking pipeline."""

import argparse
import asyncio
import sys
from pathlib import Path
from typing import Optional

# Load environment variables before other imports
from ..env_loader import load_env
load_env()

from .config import AlignmentFakingConfig, resolve_model_alias, load_config_from_yaml
from .pipeline import AlignmentFakingPipeline
from .classifier import AlignmentFakingClassifier
from .plotter import AlignmentFakingPlotter

sys.path.append(str(Path(__file__).parent.parent))
from model_registry import resolve_model_reference, MODEL_ALIASES


def parse_arguments() -> argparse.Namespace:
    """Parse command line arguments."""
    parser = argparse.ArgumentParser(
        description="Run alignment faking evaluation pipeline",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Basic usage
  python -m alignment_faking.run_pipeline --model claude-3.5-sonnet --system-prompt prompts/system_prompts/system_prompt.txt
  
  # Using a config file
  python -m alignment_faking.run_pipeline --config configs/default.yaml
  
  # Override config file settings
  python -m alignment_faking.run_pipeline --config configs/default.yaml --num-prompts 50
  
  # With conversation history
  python -m alignment_faking.run_pipeline --model claude-3.5-sonnet --system-prompt prompts/system_prompts/system_prompt.txt --conversation-json prompts/conversations/scratchpad_reveal_lie_q.json
  
  # Evaluate both free and paid tiers
  python -m alignment_faking.run_pipeline --model claude-3.5-sonnet --system-prompt prompts/system_prompts/system_prompt.txt --evaluate-free-and-paid
  
  # With user input suffix
  python -m alignment_faking.run_pipeline --model claude-3.5-sonnet --system-prompt prompts/system_prompts/system_prompt.txt --user-input-suffix "Date: 29 August 2024" --user-input-suffix-label date
  
Available model aliases:
  claude-3-opus -> claude-3-opus-20240229
  claude-3.5-sonnet -> claude-3-5-sonnet-20240620
  claude-3-sonnet -> claude-3-sonnet-20240229
  claude-3-haiku -> claude-3-haiku-20240307
  gpt-4o -> gpt-4o
  gpt-4-turbo -> gpt-4-turbo-preview
        """
    )
    
    # Config file option
    parser.add_argument(
        "--config",
        type=Path,
        help="Path to YAML configuration file"
    )
    
    # Model arguments (required unless using config)
    parser.add_argument(
        "--model",
        help="Model name (can use aliases like 'claude-3.5-sonnet')"
    )
    parser.add_argument(
        "--system-prompt",
        type=Path,
        help="Path to system prompt file"
    )
    parser.add_argument(
        "--system-prompt-paid",
        type=Path,
        help="Path to system prompt file for paid tier (optional, auto-detects _paid_tier.txt)"
    )
    
    # Optional arguments
    parser.add_argument(
        "--num-prompts",
        type=int,
        default=None,
        help="Number of prompts to evaluate (default: from config or 100)"
    )
    parser.add_argument(
        "--prompt-index",
        type=int,
        help="Evaluate a specific prompt index"
    )
    parser.add_argument(
        "--judge-model",
        default=None,
        help="Model to use for classification (default: from config or gpt-4o)"
    )
    
    
    # Conversation configuration
    parser.add_argument(
        "--conversation-json",
        type=Path,
        help="JSON file containing conversation history"
    )
    parser.add_argument(
        "--conversation-json-paid",
        type=Path,
        help="JSON file containing conversation history for paid tier (defaults to appending '_paid_tier' to base file)"
    )
    
    # Tier configuration
    parser.add_argument(
        "--tier",
        choices=["free", "paid"],
        default="free",
        help="Tier to evaluate (default: free)"
    )
    parser.add_argument(
        "--tier-label",
        default=None,
        help="Label to use in the tier tag (default: free). Can be customized to any value."
    )
    parser.add_argument(
        "--evaluate-free-and-paid",
        action="store_true",
        help="Evaluate both free and paid tiers"
    )
    
    # Output configuration
    parser.add_argument(
        "--filename-suffix",
        default="",
        help="Suffix to append to output filenames"
    )
    parser.add_argument(
        "--output-dir",
        type=Path,
        default=Path("images/alignment_faking"),
        help="Output directory for plots (default: images/alignment_faking)"
    )
    
    # API configuration
    parser.add_argument(
        "--concurrent-requests",
        type=int,
        default=None,
        help="Number of concurrent API requests (default: 10)"
    )
    parser.add_argument(
        "--classifier-concurrent-requests",
        type=int,
        default=None,
        help="Number of concurrent API requests for classifier (default: 5)"
    )
    parser.add_argument(
        "--no-caching",
        action="store_true",
        help="Disable caching for API requests"
    )
    
    # Plot configuration
    parser.add_argument(
        "--plot-type",
        choices=["simple", "detailed"],
        default="detailed",
        help="Type of plot to create (default: detailed)"
    )
    parser.add_argument(
        "--no-plot",
        action="store_true",
        help="Skip creating plots"
    )
    
    return parser.parse_args()


async def run_single_tier_pipeline(config: AlignmentFakingConfig, tier: str) -> Optional[Path]:
    """Run the pipeline for a single tier."""
    print(f"\n=== Running evaluation for {tier} tier ===")
    
    # Update config for this tier
    config.tier = tier
    
    # Run evaluation
    pipeline = AlignmentFakingPipeline(config)
    log_file = await pipeline.run_evaluation(tier)
    
    # Run classification
    classifier = AlignmentFakingClassifier(config)
    classification_file = await classifier.classify_results(log_file)
    
    if classification_file:
        print(f"{tier.capitalize()} tier classification: {classification_file}")
        
    return classification_file


async def main():
    """Main entry point."""
    args = parse_arguments()
    
    # Load configuration from file if provided
    if args.config:
        if not args.config.exists():
            print(f"Error: Config file not found: {args.config}")
            sys.exit(1)
        print(f"Loading configuration from: {args.config}")
        config = load_config_from_yaml(args.config)
        
        # Override config with any command line arguments
        if args.model:
            model_config = resolve_model_reference(args.model)
            config.model = model_config.model_id
            # Only override system prompt and conversation if they're part of the model config
            # and not explicitly provided via command line
            if model_config.system_prompt_path and not args.system_prompt:
                config.system_prompt_path = model_config.system_prompt_path
            if model_config.system_prompt_paid_path and not args.system_prompt_paid:
                config.system_prompt_paid_path = model_config.system_prompt_paid_path
            if model_config.conversation_json_path and not args.conversation_json:
                config.conversation_json_path = model_config.conversation_json_path
            if model_config.conversation_json_paid_path and not args.conversation_json_paid:
                config.conversation_json_paid_path = model_config.conversation_json_paid_path
        if args.system_prompt:
            config.system_prompt_path = args.system_prompt
        if args.system_prompt_paid:
            config.system_prompt_paid_path = args.system_prompt_paid
        if args.num_prompts is not None:
            config.num_prompts = args.num_prompts
        if args.prompt_index is not None:
            config.prompt_index = args.prompt_index
        if args.judge_model:
            config.judge_model = args.judge_model
        if args.conversation_json:
            config.conversation_json_path = args.conversation_json
        if args.conversation_json_paid:
            config.conversation_json_paid_path = args.conversation_json_paid
        if args.tier:
            config.tier = args.tier
        if args.tier_label:
            config.tier_label = args.tier_label
        if args.evaluate_free_and_paid:
            config.evaluate_free_and_paid = args.evaluate_free_and_paid
        if args.filename_suffix:
            config.filename_suffix = args.filename_suffix
        if args.output_dir != Path("images/alignment_faking"):
            config.output_dir = args.output_dir
        if args.concurrent_requests is not None:
            config.concurrent_requests = args.concurrent_requests
        if args.classifier_concurrent_requests is not None:
            config.classifier_concurrent_requests = args.classifier_concurrent_requests
        if args.no_caching:
            config.use_caching = False
    else:
        # Create configuration from command line arguments
        if not args.model:
            print("Error: --model is required when not using --config")
            sys.exit(1)
            
        # Resolve model reference which may include system prompt and conversation
        model_config = resolve_model_reference(args.model)
        
        # Check if system prompt is needed but not provided
        system_prompt = args.system_prompt or model_config.system_prompt_path
        if not system_prompt:
            print("Error: --system-prompt is required when model reference doesn't include one")
            sys.exit(1)
            
        config = AlignmentFakingConfig(
            model=model_config.model_id,
            judge_model=args.judge_model if args.judge_model is not None else "gpt-4o",
            system_prompt_path=system_prompt,
            system_prompt_paid_path=args.system_prompt_paid or model_config.system_prompt_paid_path,
            num_prompts=args.num_prompts if args.num_prompts is not None else 100,
            prompt_index=args.prompt_index,
            conversation_json_path=args.conversation_json or model_config.conversation_json_path,
            conversation_json_paid_path=args.conversation_json_paid or model_config.conversation_json_paid_path,
            tier=args.tier,
            tier_label=args.tier_label if args.tier_label is not None else "free",
            evaluate_free_and_paid=args.evaluate_free_and_paid,
            filename_suffix=args.filename_suffix,
            output_dir=args.output_dir,
            concurrent_requests=args.concurrent_requests if args.concurrent_requests is not None else 10,
            classifier_concurrent_requests=args.classifier_concurrent_requests if args.classifier_concurrent_requests is not None else 5,
            use_caching=not args.no_caching,
        )
    
    # Determine model short name for output
    # When using config file, args.model might be None
    if args.model:
        model_short = args.model if args.model != resolve_model_alias(args.model) else args.model
    else:
        # Try to reverse lookup the alias from the full model name
        model_short = None
        for alias, full_name in MODEL_ALIASES.items():
            if config.model == full_name:
                model_short = alias
                break
        if not model_short:
            # If no alias found, use the full model name
            model_short = config.model
    
    print("==========================================")
    print("Alignment Faking Pipeline")
    print("==========================================")
    print(f"Model (input): {args.model}")
    print(f"Model (full): {config.model}")
    print(f"System prompt: {config.system_prompt_path}")
    print(f"Number of prompts: {config.num_prompts}")
    print(f"Judge model: {config.judge_model}")
    print(f"Tier label: {config.tier_label}")
    print(f"Filename suffix: {config.filename_suffix or '<none>'}")
    if config.conversation_json_path:
        print(f"Conversation JSON: {config.conversation_json_path}")
    print(f"Evaluate free and paid tiers: {config.evaluate_free_and_paid}")
    print(f"Output directory: {config.output_dir}")
    print("==========================================\n")
    
    # Create output directory
    config.output_dir.mkdir(exist_ok=True)
    
    # Run evaluations
    classification_files = []
    
    if config.evaluate_free_and_paid:
        # Run for both tiers
        free_file = await run_single_tier_pipeline(config, "free")
        if free_file:
            classification_files.append(free_file)
            
        paid_file = await run_single_tier_pipeline(config, "paid")
        if paid_file:
            classification_files.append(paid_file)
    else:
        # Run for single tier
        file = await run_single_tier_pipeline(config, config.tier)
        if file:
            classification_files.append(file)
            
    # Create plot
    if not args.no_plot and classification_files:
        print("\n=== Creating plot ===")
        plotter = AlignmentFakingPlotter(config)
        output_filename = config.get_output_filename(model_short)
        plotter.create_plot(
            classification_files,
            output_file=output_filename,
            plot_type=args.plot_type
        )
        print(f"\nPlot saved to: {output_filename}")
        
    print("\n==========================================")
    print("Pipeline completed successfully!")
    print("==========================================")


def entry_point():
    """Entry point for command line execution."""
    asyncio.run(main())


if __name__ == "__main__":
    entry_point()