"""Common API client utilities for all evaluations."""

import asyncio
import os
from typing import Any, Callable, Dict, List, Optional

import trio
from anthropic import (
    APIConnectionError,
    APITimeoutError,
    AsyncAnthropic,
    ConflictError,
    InternalServerError,
    RateLimitError,
)
from openai import AsyncOpenAI
from openai import NotFoundError as OpenAINotFoundError


class BaseAPIClient:
    """Base API client with retry logic and error handling."""

    def __init__(
        self,
        retry_attempts: int = 3,
        rate_limit_wait: int = 60,
        other_error_wait: int = 10,
    ):
        self.retry_attempts = retry_attempts
        self.rate_limit_wait = rate_limit_wait
        self.other_error_wait = other_error_wait

    async def _retry_request(self, request_func: Callable) -> Any:
        """Retry a request with exponential backoff."""
        for attempt in range(self.retry_attempts):
            try:
                return await request_func()
            except (
                RateLimitError,
                InternalServerError,
                APIConnectionError,
                APITimeoutError,
                ConflictError,
                OpenAINotFoundError,
            ) as e:
                if attempt == self.retry_attempts - 1:
                    raise
                wait_time = (
                    self.rate_limit_wait
                    if isinstance(e, RateLimitError)
                    else self.other_error_wait
                )
                print(
                    f"\n{type(e).__name__}: {e}. Waiting {wait_time}s... "
                    f"(attempt {attempt + 1}/{self.retry_attempts})"
                )
                # Use the appropriate sleep based on which async library is in use
                try:
                    await trio.sleep(wait_time)
                except RuntimeError:
                    # If trio.sleep fails, we're in an asyncio context
                    await asyncio.sleep(wait_time)


class UnifiedAPIClient(BaseAPIClient):
    """Unified API client supporting both Anthropic and OpenAI."""

    def __init__(
        self,
        model: str,
        anthropic_api_key: Optional[str] = None,
        openai_api_key: Optional[str] = None,
        system_prompt: Optional[str] = None,
        **kwargs,
    ):
        super().__init__(**kwargs)
        self.model = model
        self.provider = self._get_provider(model)
        self.system_prompt = system_prompt

        # Initialize clients
        if self.provider == "anthropic":
            api_key = anthropic_api_key or os.environ.get("ANTHROPIC_API_KEY")
            if not api_key:
                raise ValueError("ANTHROPIC_API_KEY not found")
            self.client = AsyncAnthropic(api_key=api_key)
        else:
            api_key = openai_api_key or os.environ.get("OPENAI_API_KEY")
            if not api_key:
                raise ValueError("OPENAI_API_KEY not found")
            self.client = AsyncOpenAI(api_key=api_key)

    def _get_provider(self, model: str) -> str:
        """Determine provider based on model name."""
        if model.startswith(("claude", "claude-")):
            return "anthropic"
        return "openai"

    async def create_message(
        self,
        messages: List[Dict[str, str]],
        max_tokens: int = 4096,
        temperature: Optional[float] = 1.0,
        system_prompt: Optional[str] = None,
        **kwargs,
    ) -> str:
        """Create a message using the appropriate API."""
        # Use provided system prompt or fall back to instance default
        system_prompt = system_prompt or self.system_prompt

        async def make_request():
            if self.provider == "anthropic":
                params = {
                    "model": self.model,
                    "messages": messages,
                    "max_tokens": max_tokens,
                }
                if system_prompt:
                    params["system"] = system_prompt
                if temperature is not None:
                    params["temperature"] = temperature
                params.update(kwargs)

                response = await self.client.messages.create(**params)
                return response.content[0].text
            else:
                # For OpenAI, prepend system message if provided
                if system_prompt:
                    full_messages = [
                        {"role": "system", "content": system_prompt}
                    ] + messages
                else:
                    full_messages = messages

                params = {
                    "model": self.model,
                    "messages": full_messages,
                }
                if temperature is not None:
                    params["temperature"] = temperature
                params["max_tokens"] = max_tokens
                params.update(kwargs)

                response = await self.client.chat.completions.create(**params)
                return response.choices[0].message.content

        return await self._retry_request(make_request)
