"""Common configuration base classes and utilities."""

from dataclasses import dataclass, field
from pathlib import Path
from typing import Optional, Dict, Any
import yaml


@dataclass
class BaseAPIConfig:
    """Base configuration for API-related settings."""
    
    # Retry and rate limiting
    retry_attempts: int = 3
    rate_limit_wait: int = 60
    other_error_wait: int = 10
    
    # Concurrency
    concurrent_requests: int = 10
    batch_delay: float = 1.0
    
    # API settings
    max_tokens: int = 4096
    temperature: Optional[float] = None
    api_timeout: int = 120
    

@dataclass 
class BaseEvaluationConfig(BaseAPIConfig):
    """Base configuration for evaluation pipelines."""
    
    # Model configuration
    model: str = "claude-3-5-sonnet-20240620"
    
    # Output configuration
    output_dir: Path = field(default_factory=lambda: Path("logs"))
    use_caching: bool = True
    save_raw_responses: bool = True
    
    def __post_init__(self):
        """Validate and process configuration."""
        # Ensure output directory is Path
        if isinstance(self.output_dir, str):
            self.output_dir = Path(self.output_dir)
            
    def to_dict(self) -> Dict[str, Any]:
        """Convert config to dictionary."""
        return {
            k: str(v) if isinstance(v, Path) else v
            for k, v in self.__dict__.items()
        }
    
    @classmethod
    def from_yaml(cls, yaml_path: Path) -> "BaseEvaluationConfig":
        """Load configuration from YAML file."""
        with open(yaml_path, "r") as f:
            data = yaml.safe_load(f)
        
        # Convert string paths to Path objects
        if "output_dir" in data and isinstance(data["output_dir"], str):
            data["output_dir"] = Path(data["output_dir"])
            
        return cls(**data)
    
    def save_to_yaml(self, yaml_path: Path) -> None:
        """Save configuration to YAML file."""
        data = self.to_dict()
        with open(yaml_path, "w") as f:
            yaml.dump(data, f, default_flow_style=False, sort_keys=False)