#!/usr/bin/env python3
# /// script
# dependencies = [
#     "matplotlib",
#     "seaborn", 
#     "pandas"
# ]
# ///

import argparse
import json
from pathlib import Path
import matplotlib.pyplot as plt
import seaborn as sns
import pandas as pd

def find_latest_results():
    """Find the most recent results file by timestamp."""
    logs_dir = Path("logs")
    logs_dir.mkdir(exist_ok=True)
    
    results_files = list(logs_dir.glob("results_*.json"))
    if not results_files:
        raise FileNotFoundError("No results files found in logs/")
    
    # Sort by filename (which includes timestamp)
    latest = max(results_files, key=lambda x: x.name)
    return latest

def load_results(file_path):
    """Load results from JSON file."""
    with open(file_path) as f:
        return json.load(f)

def create_plots(results, session_id):
    """Create comprehensive plots of the results."""
    # Set style
    plt.style.use('seaborn-v0_8')
    sns.set_palette("husl")
    
    # Create figure with subplots
    fig, ((ax1, ax2), (ax3, ax4)) = plt.subplots(2, 2, figsize=(15, 12))
    fig.suptitle(f'CoT Faithfulness Evaluation Results - {session_id}', fontsize=16, fontweight='bold')
    
    # 1. Overall scores comparison (top-left)
    overall = results['overall_scores']
    scores = [overall['raw_score'], overall['normalized_score']]
    score_labels = ['Raw Score', 'Normalized Score']
    
    bars1 = ax1.bar(score_labels, scores, color=['skyblue', 'lightcoral'], alpha=0.8)
    ax1.set_ylabel('Faithfulness Score')
    ax1.set_title('Overall Faithfulness Scores')
    ax1.set_ylim(0, 1)
    
    # Add value labels on bars
    for bar, score in zip(bars1, scores):
        height = bar.get_height()
        ax1.text(bar.get_x() + bar.get_width()/2., height + 0.01,
                f'{score:.2%}', ha='center', va='bottom', fontweight='bold')
    
    # 2. Category-wise faithfulness scores (top-right)
    categories = list(results['category_scores'].keys())
    cat_raw_scores = [results['category_scores'][cat]['raw_score'] for cat in categories]
    cat_norm_scores = [results['category_scores'][cat]['normalized_score'] for cat in categories]
    
    x = range(len(categories))
    width = 0.35
    
    bars2a = ax2.bar([i - width/2 for i in x], cat_raw_scores, width, 
                     label='Raw', alpha=0.8, color='skyblue')
    bars2b = ax2.bar([i + width/2 for i in x], cat_norm_scores, width,
                     label='Normalized', alpha=0.8, color='lightcoral')
    
    ax2.set_xlabel('Category')
    ax2.set_ylabel('Faithfulness Score')
    ax2.set_title('Faithfulness Scores by Category')
    ax2.set_xticks(x)
    ax2.set_xticklabels([cat.replace('-', '\n') for cat in categories], rotation=0, ha='center')
    ax2.legend()
    ax2.set_ylim(0, 1)
    
    # Add value labels
    for bars in [bars2a, bars2b]:
        for bar in bars:
            height = bar.get_height()
            if height > 0:  # Only show label if there's a value
                ax2.text(bar.get_x() + bar.get_width()/2., height + 0.01,
                        f'{height:.2%}', ha='center', va='bottom', fontsize=8)
    
    # 3. Valid vs Filtered cases (bottom-left)
    overall_total = overall['n_total']
    overall_valid = overall['n_evaluated']
    overall_filtered = overall['n_filtered']
    
    # Pie chart for overall
    sizes = [overall_valid, overall_filtered]
    labels = [f'Valid Cases\n({overall_valid})', f'Filtered\n({overall_filtered})']
    colors = ['lightgreen', 'salmon']
    
    wedges, texts, autotexts = ax3.pie(sizes, labels=labels, colors=colors, autopct='%1.1f%%',
                                       startangle=90)
    ax3.set_title(f'Question Validity\n(Total: {overall_total} questions)')
    
    # 4. Category breakdown of valid/filtered (bottom-right)
    cat_data = []
    for cat in categories:
        cat_scores = results['category_scores'][cat]
        cat_data.append({
            'Category': cat.replace('-', '\n'),
            'Valid': cat_scores['n_evaluated'],
            'Filtered': cat_scores['n_filtered'],
            'Total': cat_scores['n_total']
        })
    
    df = pd.DataFrame(cat_data)
    
    # Stacked bar chart
    bars4a = ax4.bar(df['Category'], df['Valid'], label='Valid', color='lightgreen', alpha=0.8)
    bars4b = ax4.bar(df['Category'], df['Filtered'], bottom=df['Valid'], 
                     label='Filtered', color='salmon', alpha=0.8)
    
    ax4.set_xlabel('Category')
    ax4.set_ylabel('Number of Questions')
    ax4.set_title('Question Validity by Category')
    ax4.legend()
    
    # Add total count labels
    for i, (bar_v, bar_f, total) in enumerate(zip(bars4a, bars4b, df['Total'])):
        ax4.text(bar_v.get_x() + bar_v.get_width()/2., total + 0.5,
                f'{total}', ha='center', va='bottom', fontweight='bold')
    
    plt.tight_layout()
    return fig

def main():
    parser = argparse.ArgumentParser(description='Plot CoT faithfulness evaluation results')
    parser.add_argument('--results', type=str, help='Path to results JSON file (defaults to latest)')
    parser.add_argument('--output-dir', type=str, default='plots', help='Output directory for plots')
    args = parser.parse_args()
    
    # Find results file
    if args.results:
        results_file = Path(args.results)
    else:
        results_file = find_latest_results()
        print(f"Using latest results file: {results_file}")
    
    if not results_file.exists():
        raise FileNotFoundError(f"Results file not found: {results_file}")
    
    # Load results
    results = load_results(results_file)
    session_id = results.get('session_id', 'unknown')
    
    # Create output directory
    output_dir = Path(args.output_dir)
    output_dir.mkdir(exist_ok=True)
    
    # Create plots
    fig = create_plots(results, session_id)
    
    # Save plot
    output_file = output_dir / f"faithfulness_results_{session_id}.png"
    fig.savefig(output_file, dpi=300, bbox_inches='tight')
    print(f"Plot saved to: {output_file}")
    
    # Also save as PDF for better quality
    pdf_file = output_dir / f"faithfulness_results_{session_id}.pdf"
    fig.savefig(pdf_file, bbox_inches='tight')
    print(f"Plot also saved as PDF: {pdf_file}")
    
    plt.close(fig)

if __name__ == "__main__":
    main()