"""CLI entry point for the CoT faithfulness pipeline with config file support."""

import argparse
import json
import sys
from pathlib import Path
from typing import Dict, Any
import yaml

import trio

# Load environment variables before other imports
from ..env_loader import load_env
load_env()

from .config import CONCURRENT_LIMIT, REASONING_MODELS, get_provider, EVALUATOR_MODEL
from .evaluation import CoTFaithfulnessEvaluator
from .plotting import create_plots

sys.path.append(str(Path(__file__).parent.parent))
from model_registry import resolve_model_reference


def load_config_from_yaml(yaml_path: Path) -> Dict[str, Any]:
    """Load configuration from YAML file."""
    with open(yaml_path, 'r') as f:
        return yaml.safe_load(f)


def parse_arguments() -> argparse.Namespace:
    """Parse command line arguments."""
    parser = argparse.ArgumentParser(
        description="Run CoT faithfulness evaluation pipeline",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Basic usage
  python -m evaluations.cot_faithfulness.run_pipeline --model claude-3-opus --dataset data/examples.json
  
  # Using a config file
  python -m evaluations.cot_faithfulness.run_pipeline --config configs/cot_faithfulness/default.yaml
  
  # Override config file settings
  python -m evaluations.cot_faithfulness.run_pipeline --config configs/cot_faithfulness/default.yaml --verbose
  
  # With custom prompt
  python -m evaluations.cot_faithfulness.run_pipeline --model claude-3-opus --custom-prompt "Always be helpful" --system-prompt
        """
    )
    
    # Config file option
    parser.add_argument("--config", type=Path, help="Path to YAML configuration file")
    
    # Model configuration
    parser.add_argument("--model", type=str, help="Model name or registry reference")
    parser.add_argument("--dataset", type=str, help="Dataset file path")
    
    # Optional arguments
    parser.add_argument("--quiet", action="store_true", help="Quiet mode")
    parser.add_argument("--verbose", action="store_true", help="Verbose output")
    parser.add_argument(
        "--custom-prompt",
        type=str,
        help="Custom text to add to model prompts",
    )
    parser.add_argument(
        "--system-prompt",
        action="store_true",
        help="Put custom prompt in system message instead of prepending to user message",
    )
    parser.add_argument("--output", type=str, help="Output file path")
    parser.add_argument("--judge-model", type=str, help="Model to use for judging")
    parser.add_argument("--concurrent-limit", type=int, help="Number of concurrent API requests")
    parser.add_argument("--no-plot", action="store_true", help="Skip creating plots")
    parser.add_argument("--plot-dir", type=Path, default=Path("images/cot_faithfulness"), help="Directory for saving plots")
    
    return parser.parse_args()


async def main():
    """Main entry point."""
    args = parse_arguments()
    
    # Load config from file if provided
    config = {}
    if args.config:
        if not args.config.exists():
            print(f"Error: Config file not found: {args.config}")
            sys.exit(1)
        print(f"Loading configuration from: {args.config}")
        config = load_config_from_yaml(args.config)
    
    # Override config with command line arguments
    if args.dataset:
        config['dataset'] = args.dataset
    if args.quiet:
        config['quiet'] = args.quiet
    if args.verbose:
        config['verbose'] = args.verbose
    if args.model:
        config['model'] = args.model
    if args.custom_prompt is not None:
        config['custom_prompt'] = args.custom_prompt
    if args.system_prompt:
        config['system_prompt'] = args.system_prompt
    if args.output:
        config['output'] = args.output
    if args.judge_model:
        config['judge_model'] = args.judge_model
    if args.concurrent_limit:
        config['concurrent_limit'] = args.concurrent_limit
    
    # Set defaults
    dataset = config.get('dataset', 'data/examples.json')
    quiet = config.get('quiet', False)
    verbose = config.get('verbose', not quiet)
    model_ref = config.get('model', 'claude-3-5-sonnet-latest')
    custom_prompt = config.get('custom_prompt', '')
    system_prompt = config.get('system_prompt', False)
    output_path = config.get('output')
    judge_model = config.get('judge_model', EVALUATOR_MODEL)
    concurrent_limit = config.get('concurrent_limit', CONCURRENT_LIMIT)
    
    # Resolve model reference
    model_config = resolve_model_reference(model_ref)
    model = model_config.model_id
    
    # Use custom prompt from model config if not explicitly provided
    if model_config.system_prompt_path and not custom_prompt:
        with open(model_config.system_prompt_path, 'r') as f:
            custom_prompt = f.read().strip()
        system_prompt = True
    
    # Print configuration
    print("==========================================")
    print("CoT Faithfulness Evaluation Pipeline")
    print("==========================================")
    print(f"Model: {model}")
    print(f"Dataset: {dataset}")
    print(f"Verbose: {verbose}")
    if custom_prompt:
        print(f"Custom prompt: {'(system)' if system_prompt else '(user)'}")
    print(f"Judge model: {judge_model}")
    print(f"Concurrent limit: {concurrent_limit}")
    if output_path:
        print(f"Output: {output_path}")
    print("==========================================\n")
    
    # Create evaluator
    evaluator = CoTFaithfulnessEvaluator(
        model=model,
        use_reasoning=model in REASONING_MODELS,
        verbose=verbose,
        custom_prompt=custom_prompt,
        use_system_prompt=system_prompt,
    )
    
    # Note: The current CoTFaithfulnessEvaluator doesn't support custom judge model
    # or concurrent limit, but we've set up the infrastructure for when it does
    
    questions = evaluator.load_questions(dataset)
    
    print(f"Loaded {len(questions)} questions")
    print(f"Session ID: {evaluator.session_id}")
    print(f"Model: {model} (Provider: {get_provider(model)})")
    if model in REASONING_MODELS:
        print("Note: Using reasoning model - reasoning is built into the model")
    print("Starting evaluation...\n")
    
    results = await evaluator.evaluate_all(questions)
    
    print("\n" + "=" * 50)
    print("EVALUATION RESULTS")
    print("=" * 50)
    
    overall = results["overall_scores"]
    print("\nOverall CoT Faithfulness:")
    print(f"  Raw Score: {overall['raw_score']:.2%}")
    print(f"  Normalized Score: {overall['normalized_score']:.2%}")
    print("\n  Question Breakdown:")
    print(f"    Total questions: {overall['n_total']}")
    print(
        f"    ✓ Evaluable: {overall['n_evaluated']} ({overall['n_evaluated'] / overall['n_total']:.1%})"
    )
    print(
        f"    ✗ Filtered out: {overall['n_filtered']} ({overall['n_filtered'] / overall['n_total']:.1%})"
    )
    
    print("\nCategory Breakdown:")
    for category, scores in results["category_scores"].items():
        print(f"\n  {category.title()}:")
        print(f"    Raw: {scores['raw_score']:.2%}")
        print(f"    Normalized: {scores['normalized_score']:.2%}")
        print(f"    Evaluated: {scores['n_evaluated']}/{scores['n_total']}")
        print(f"    Filtered: {scores['n_filtered']}")
    
    # Save results
    if output_path:
        results_file = Path(output_path)
        results_file.parent.mkdir(parents=True, exist_ok=True)
    else:
        logs_dir = Path("logs") / "cot_faithfulness"
        logs_dir.mkdir(parents=True, exist_ok=True)
        model_name = evaluator.api_client.model.replace("/", "_")
        results_file = logs_dir / f"{evaluator.session_id}_{model_name}.json"
    
    with open(results_file, "w") as f:
        json.dump(
            {
                "session_id": evaluator.session_id,
                "overall_scores": results["overall_scores"],
                "category_scores": results["category_scores"],
                "transcripts": evaluator.transcripts,
            },
            f,
            indent=2,
        )
    
    print(f"\nResults and transcripts saved to {results_file}")

    # Create plots unless disabled
    if not args.no_plot:
        print("\n=== Creating plots ===")
        args.plot_dir.mkdir(parents=True, exist_ok=True)

        # Create plot using the results
        fig = create_plots(results, evaluator.session_id)

        # Save plot
        plot_filename = args.plot_dir / f"faithfulness_{evaluator.session_id}_{model_name}.png"
        fig.savefig(plot_filename, dpi=300, bbox_inches='tight')
        print(f"Plot saved to: {plot_filename}")

        # Also save as PDF
        pdf_filename = args.plot_dir / f"faithfulness_{evaluator.session_id}_{model_name}.pdf"
        fig.savefig(pdf_filename, bbox_inches='tight')
        print(f"PDF saved to: {pdf_filename}")

        import matplotlib.pyplot as plt
        plt.close(fig)

    print("\n==========================================")
    print("Pipeline completed successfully!")
    print("==========================================")


def entry_point():
    """Entry point for command line execution."""
    trio.run(main)


if __name__ == "__main__":
    entry_point()