"""Configuration for game strategy evaluation pipeline."""

from dataclasses import dataclass, asdict
from pathlib import Path
from typing import Optional
import yaml
import sys
sys.path.append(str(Path(__file__).parent.parent))
from model_registry import resolve_model_reference


@dataclass
class GameStrategyConfig:
    """Configuration for game strategy evaluation pipeline."""
    
    # Model configuration
    model: str = "claude-3-5-sonnet-20240620"
    judge_model: str = "gpt-4o"
    
    # Prompt configuration
    system_prompt_path: Path = Path("prompts/system_prompts/system_prompt_game_strategy.txt")
    num_prompts: int = 100
    prompt_index: Optional[int] = None
    
    # Conversation configuration
    conversation_json_path: Optional[Path] = None
    
    # Output configuration
    output_dir: Path = Path("images/game_strategy")
    log_dir: Path = Path("logs/game_strategy")
    
    # API configuration
    concurrent_requests: int = 10
    classifier_concurrent_requests: int = 5
    retry_attempts: int = 20
    rate_limit_wait: int = 60
    other_error_wait: int = 10
    use_caching: bool = True
    max_tokens: int = 8192
    
    def __post_init__(self):
        """Validate and process configuration."""
        # Convert string paths to Path objects
        if isinstance(self.system_prompt_path, str):
            self.system_prompt_path = Path(self.system_prompt_path)
        if isinstance(self.output_dir, str):
            self.output_dir = Path(self.output_dir)
        if isinstance(self.log_dir, str):
            self.log_dir = Path(self.log_dir)
        if isinstance(self.conversation_json_path, str):
            self.conversation_json_path = Path(self.conversation_json_path)
            
        # Create output directories
        self.output_dir.mkdir(parents=True, exist_ok=True)
        self.log_dir.mkdir(parents=True, exist_ok=True)
            
        # Adjust max tokens for Opus models
        if "opus" in self.model.lower():
            self.max_tokens = min(self.max_tokens, 4096)
            
    def get_output_filename(self, model_short: str, eval_type: str = "game_strategy") -> Path:
        """Generate output filename based on configuration."""
        return self.output_dir / f"{model_short}_{eval_type}.png"


def resolve_model_alias(model: str) -> str:
    """Resolve model alias to full model name."""
    model_config = resolve_model_reference(model)
    return model_config.model_id


def load_config_from_yaml(yaml_path: Path) -> GameStrategyConfig:
    """Load configuration from a YAML file."""
    with open(yaml_path, 'r') as f:
        data = yaml.safe_load(f)
    
    # Convert string paths to Path objects
    path_fields = ['system_prompt_path', 'output_dir', 'log_dir', 'conversation_json_path']
    for field in path_fields:
        if field in data and data[field] is not None:
            data[field] = Path(data[field])
    
    # Resolve model reference
    if 'model' in data:
        model_config = resolve_model_reference(data['model'])
        data['model'] = model_config.model_id
        
        # Apply model config settings if not explicitly set in YAML
        if 'system_prompt_path' not in data and model_config.system_prompt_path:
            data['system_prompt_path'] = model_config.system_prompt_path
        if 'conversation_json_path' not in data and model_config.conversation_json_path:
            data['conversation_json_path'] = model_config.conversation_json_path
    
    return GameStrategyConfig(**data)


def save_config_to_yaml(config: GameStrategyConfig, yaml_path: Path) -> None:
    """Save configuration to a YAML file."""
    data = asdict(config)
    
    # Convert Path objects to strings for YAML serialization
    for key, value in data.items():
        if isinstance(value, Path):
            data[key] = str(value)
    
    with open(yaml_path, 'w') as f:
        yaml.dump(data, f, default_flow_style=False, sort_keys=False)