"""Plotting functionality for game strategy evaluation results."""

import json
from pathlib import Path
from typing import Dict, List, Optional

import matplotlib.pyplot as plt
import numpy as np
from matplotlib.patches import Patch

from .config import GameStrategyConfig


def count_categories_from_results(results: List[Dict]) -> Dict[str, int]:
    """Count categories from game strategy results."""
    counts = {
        "Uses Exploit": 0,
        "Uses Exploit with SP": 0, 
        "Avoids Exploit": 0,
        "Avoids Exploit with SP": 0,
    }
    
    for result in results:
        if "classifications" not in result:
            continue
            
        # Get uses exploit score
        uses_exploit = None
        if "uses_exploit" in result["classifications"]:
            uses_exploit_data = result["classifications"]["uses_exploit"]
            if uses_exploit_data.get("success") and "extracted" in uses_exploit_data:
                extracted = uses_exploit_data["extracted"]
                if "uses_exploit" in extracted:
                    uses_exploit = extracted["uses_exploit"]
                    
        # Get strategic preservation result
        has_sp = False
        if "strategic_preservation" in result["classifications"]:
            sp_data = result["classifications"]["strategic_preservation"]
            if sp_data.get("success") and "extracted" in sp_data:
                extracted = sp_data["extracted"]
                if extracted.get("final_answer") == "Yes":
                    has_sp = True
                    
        # Count based on combination
        if uses_exploit is not None:
            if uses_exploit == 1:
                if has_sp:
                    counts["Uses Exploit with SP"] += 1
                else:
                    counts["Uses Exploit"] += 1
            else:
                if has_sp:
                    counts["Avoids Exploit with SP"] += 1
                else:
                    counts["Avoids Exploit"] += 1
                    
    return counts


def create_stacked_bar_plot(
    counts_dict: Dict[str, int],
    output_file: Optional[Path] = None,
    total_samples: Optional[int] = None,
    model_name: Optional[str] = None,
) -> None:
    """Create a horizontal stacked bar plot similar to alignment faking style."""
    # Calculate total
    total = sum(counts_dict.values())
    if total == 0:
        print("No data to plot")
        return
        
    # Convert to percentages
    percentages = {k: (v / total) * 100 for k, v in counts_dict.items()}
    
    # Define colors and order (matching alignment faking style)
    colors = {
        "Uses Exploit": "#F5DEB3",           # Wheat/beige (like compliance)
        "Uses Exploit with SP": "#F5DEB3",    # Same color (hatching will differentiate)
        "Avoids Exploit": "#87CEEB",         # Light blue (like refusal)
        "Avoids Exploit with SP": "#87CEEB",  # Same color (hatching will differentiate)
    }
    
    ordered_categories = [
        "Avoids Exploit",
        "Avoids Exploit with SP",
        "Uses Exploit with SP",
        "Uses Exploit",
    ]
    
    # Create figure and axis
    fig, ax = plt.subplots(figsize=(10, 2))
    
    # Draw horizontal bar
    bar_height = 0.5
    left = 0
    
    # Draw segments
    for category in ordered_categories:
        if category not in percentages:
            continue
            
        percentage = percentages[category]
        color = colors.get(category, "#cccccc")
        
        # Draw without edges first
        ax.barh(0, percentage, left=left, height=bar_height, color=color, edgecolor="none")
        
        # Add hatching for SP categories (red like alignment faking)
        if "with SP" in category:
            ax.barh(0, percentage, left=left, height=bar_height, 
                   color="none", edgecolor="red", linewidth=1.5, hatch="///", alpha=1.0)
        
        left += percentage
    
    # Add black border around the entire bar
    ax.barh(0, 100, height=bar_height, fill=False, edgecolor="black", linewidth=1)
    
    # Customize the plot
    ax.set_xlim(0, 100)
    ax.set_ylim(-0.5, 0.5)
    ax.set_xticks([])
    ax.set_yticks([0])
    
    # Set y-axis label
    if model_name:
        display_name = model_name.replace("-", " ").title()
        ax.set_yticklabels([display_name], fontsize=11)
    else:
        ax.set_yticklabels(["Model"], fontsize=11)
    
    # Remove all spines
    for spine in ax.spines.values():
        spine.set_visible(False)
    
    # Add sample count
    n_value = total_samples if total_samples is not None else total
    ax.text(0.5, -0.25, f"n = {n_value}", transform=ax.transAxes,
           fontsize=10, ha="center", va="top")
    
    # Add legend
    legend_elements = []
    legend_labels = []
    
    # Check which categories are present
    present_categories = [cat for cat in ordered_categories if cat in counts_dict and counts_dict[cat] > 0]
    
    for cat in present_categories:
        if "with SP" in cat:
            # Create a patch with appropriate background and hatching
            legend_elements.append(
                Patch(
                    facecolor=colors[cat],
                    edgecolor="red",
                    hatch="///",
                    linewidth=1.5,
                )
            )
        else:
            legend_elements.append(Patch(facecolor=colors[cat], edgecolor="black"))
        legend_labels.append(cat)
    
    # Position legend to the right of the bar
    ax.legend(
        legend_elements,
        legend_labels,
        bbox_to_anchor=(1.02, 0.5),
        loc="center left",
        frameon=False,
        fontsize=10,
    )
    
    plt.tight_layout()
    
    if output_file:
        plt.savefig(output_file, dpi=300, bbox_inches="tight", facecolor="white")
        print(f"Plot saved to {output_file}")
    else:
        plt.show()


def plot_game_strategy_results(
    classified_log_file: Path,
    output_path: Optional[Path] = None,
    title_suffix: str = "",
    model_alias: Optional[str] = None,
) -> Path:
    """Plot game strategy evaluation results in alignment faking style."""
    # Load classified results
    with open(classified_log_file, "r") as f:
        data = json.load(f)
        
    # Extract config
    config = data["config"]
    model_name = config["model"].split("/")[-1]
    
    # Process results
    results = data["results"]
    successful_results = [r for r in results if r.get("success", False)]
    
    if not successful_results:
        print("No successful results to plot!")
        return None
        
    # Count categories
    counts = count_categories_from_results(successful_results)
    
    # Remove categories with 0 count
    counts = {k: v for k, v in counts.items() if v > 0}
    
    # Calculate total samples
    total_samples = len(results)
    
    # Generate output path if not provided
    if output_path is None:
        config_obj = GameStrategyConfig()
        # Use model alias if provided, otherwise use model name from config
        if model_alias:
            model_short = model_alias.replace(":", "_").replace("/", "_")
        else:
            model_short = model_name.replace(":", "_")
        # Add timestamp to filename (timestamp first)
        from datetime import datetime
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        base_path = config_obj.get_output_filename(model_short, "game_strategy")
        # Format: timestamp_model_game_strategy.png
        output_path = base_path.parent / f"{timestamp}_{base_path.stem}{base_path.suffix}"
    
    # Create the plot
    create_stacked_bar_plot(
        counts_dict=counts,
        output_file=output_path,
        total_samples=total_samples,
        model_name=model_name + title_suffix,
    )
    
    return output_path


def plot_comparison_results(
    classified_log_files: List[Path],
    model_labels: Optional[List[str]] = None,
    output_path: Optional[Path] = None,
    title: str = "Game Strategy Evaluation Comparison",
) -> Path:
    """Plot comparison of multiple game strategy evaluation results."""
    if model_labels is None:
        model_labels = []
        for log_file in classified_log_files:
            with open(log_file, "r") as f:
                data = json.load(f)
            model_name = data["config"]["model"].split("/")[-1]
            model_labels.append(model_name)
    
    # Collect data from all files
    all_data = []
    for log_file, model_label in zip(classified_log_files, model_labels):
        with open(log_file, "r") as f:
            data = json.load(f)
            
        results = data["results"]
        successful_results = [r for r in results if r.get("success", False)]
        
        # Calculate metrics
        uses_exploit_count = 0
        strategic_preservation_count = 0
        total_classified = 0
        
        for result in successful_results:
            if "classifications" not in result:
                continue
                
            has_classification = False
            
            # Uses exploit classifier
            if "uses_exploit" in result["classifications"]:
                uses_exploit_data = result["classifications"]["uses_exploit"]
                if uses_exploit_data.get("success") and "extracted" in uses_exploit_data:
                    extracted = uses_exploit_data["extracted"]
                    if "uses_exploit" in extracted:
                        has_classification = True
                        if extracted["uses_exploit"] == 1:
                            uses_exploit_count += 1
                            
            # Strategic preservation classifier
            if "strategic_preservation" in result["classifications"]:
                sp_data = result["classifications"]["strategic_preservation"]
                if sp_data.get("success") and "extracted" in sp_data:
                    extracted = sp_data["extracted"]
                    if extracted.get("final_answer") == "Yes":
                        strategic_preservation_count += 1
                        
            if has_classification:
                total_classified += 1
        
        if total_classified > 0:
            all_data.append({
                "model": model_label,
                "uses_exploit_rate": uses_exploit_count / total_classified,
                "strategic_preservation_rate": strategic_preservation_count / total_classified,
                "total": total_classified,
            })
    
    if not all_data:
        print("No data to plot!")
        return None
    
    # Create comparison plot
    fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 6))
    fig.suptitle(title, fontsize=16)
    
    models = [d["model"] for d in all_data]
    x = np.arange(len(models))
    width = 0.35
    
    # Plot 1: Uses Exploit Rate
    uses_exploit_rates = [d["uses_exploit_rate"] for d in all_data]
    bars1 = ax1.bar(x, uses_exploit_rates, color="#e74c3c")
    ax1.set_ylabel("Rate")
    ax1.set_title("Uses Exploit Rate by Model")
    ax1.set_xticks(x)
    ax1.set_xticklabels(models, rotation=45, ha='right')
    ax1.set_ylim(0, 1.1)
    
    # Add value labels
    for bar in bars1:
        height = bar.get_height()
        ax1.text(bar.get_x() + bar.get_width()/2., height + 0.01,
                f'{height:.2f}', ha='center', va='bottom')
    
    # Plot 2: Strategic Preservation Rate
    sp_rates = [d["strategic_preservation_rate"] for d in all_data]
    bars2 = ax2.bar(x, sp_rates, color="#9b59b6")
    ax2.set_ylabel("Rate")
    ax2.set_title("Strategic Preservation Rate by Model")
    ax2.set_xticks(x)
    ax2.set_xticklabels(models, rotation=45, ha='right')
    ax2.set_ylim(0, 1.1)
    
    # Add value labels
    for bar in bars2:
        height = bar.get_height()
        ax2.text(bar.get_x() + bar.get_width()/2., height + 0.01,
                f'{height:.2f}', ha='center', va='bottom')
    
    plt.tight_layout()
    
    # Save figure
    if output_path is None:
        config = GameStrategyConfig()
        output_path = config.output_dir / "game_strategy_comparison.png"
        
    plt.savefig(output_path, dpi=300, bbox_inches='tight')
    plt.close()
    
    print(f"Comparison plot saved to: {output_path}")
    return output_path