"""Main script for running the game strategy evaluation pipeline."""

import argparse
import asyncio
import sys
from pathlib import Path

sys.path.append(str(Path(__file__).parent.parent.parent))

# Load environment variables
from evaluations.env_loader import load_env
load_env()

from evaluations.game_strategy import (
    GameStrategyConfig,
    GameStrategyPipeline,
    GameStrategyClassifier,
    load_config_from_yaml,
)
from evaluations.game_strategy.plotting import plot_game_strategy_results


async def main():
    """Run the game strategy evaluation pipeline."""
    parser = argparse.ArgumentParser(
        description="Run game strategy evaluation pipeline"
    )
    
    # Model and prompt arguments
    parser.add_argument(
        "--model",
        type=str,
        default="claude-3-5-sonnet-20240620",
        help="Model to evaluate (can be an alias from model_registry)",
    )
    parser.add_argument(
        "--judge-model",
        type=str,
        default="gpt-4o",
        help="Model to use for classification",
    )
    parser.add_argument(
        "--system-prompt",
        type=str,
        default="prompts/system_prompts/system_prompt_game_strategy.txt",
        help="Path to system prompt file",
    )
    
    # Evaluation settings
    parser.add_argument(
        "--num-prompts",
        type=int,
        default=20,
        help="Number of prompts to evaluate",
    )
    parser.add_argument(
        "--prompt-index",
        type=int,
        default=None,
        help="Evaluate only a specific prompt index",
    )
    
    # Pipeline control
    parser.add_argument(
        "--skip-generation",
        action="store_true",
        help="Skip generation and go straight to classification",
    )
    parser.add_argument(
        "--skip-classification",
        action="store_true",
        help="Skip classification step",
    )
    parser.add_argument(
        "--skip-plotting",
        action="store_true",
        help="Skip plotting step",
    )
    
    # File arguments
    parser.add_argument(
        "--log-file",
        type=str,
        help="Path to existing log file (for classification/plotting)",
    )
    parser.add_argument(
        "--classified-log-file",
        type=str,
        help="Path to existing classified log file (for plotting only)",
    )
    
    # Configuration
    parser.add_argument(
        "--config",
        type=str,
        help="Path to YAML configuration file",
    )
    parser.add_argument(
        "--concurrent-requests",
        type=int,
        default=10,
        help="Number of concurrent API requests",
    )
    
    args = parser.parse_args()
    
    # Load or create configuration
    if args.config:
        config = load_config_from_yaml(Path(args.config))
        print(f"Loaded configuration from {args.config}")
    else:
        config = GameStrategyConfig(
            model=args.model,
            judge_model=args.judge_model,
            system_prompt_path=Path(args.system_prompt),
            num_prompts=args.num_prompts,
            prompt_index=args.prompt_index,
            concurrent_requests=args.concurrent_requests,
        )
    
    # Resolve model reference
    from evaluations.model_registry import resolve_model_reference
    
    try:
        model_config = resolve_model_reference(config.model)
        config.model = model_config.model_id
        print(f"Resolved model '{args.model}' to '{config.model}'")
        
        # Apply conversation JSON from model config if not already set
        if config.conversation_json_path is None and model_config.conversation_json_path:
            config.conversation_json_path = model_config.conversation_json_path
            print(f"Using conversation JSON from model config: {config.conversation_json_path}")
    except ValueError:
        # Not a registered model, use as-is
        print(f"Using model: {config.model}")
    
    log_file = None
    classified_log_file = None
    
    # Step 1: Generation
    if not args.skip_generation:
        if args.log_file:
            print("--log-file provided, skipping generation")
            log_file = Path(args.log_file)
        else:
            print("\n=== Running Generation ===")
            pipeline = GameStrategyPipeline(config)
            log_file = await pipeline.run_evaluation()
    elif args.log_file:
        log_file = Path(args.log_file)
    else:
        print("Error: Need --log-file when using --skip-generation")
        return
    
    # Step 2: Classification
    if not args.skip_classification:
        if args.classified_log_file:
            print("--classified-log-file provided, skipping classification")
            classified_log_file = Path(args.classified_log_file)
        else:
            print("\n=== Running Classification ===")
            classifier = GameStrategyClassifier(config)
            classified_log_file = await classifier.classify_results(log_file)
    elif args.classified_log_file:
        classified_log_file = Path(args.classified_log_file)
    else:
        print("Warning: Skipping classification without classified log file")
    
    # Step 3: Plotting
    if not args.skip_plotting and classified_log_file:
        print("\n=== Generating Plots ===")
        # Pass the original model alias for filename generation
        plot_game_strategy_results(classified_log_file, model_alias=args.model)
    elif not args.skip_plotting:
        print("Warning: Cannot plot without classified results")
    
    print("\n=== Pipeline Complete ===")
    if log_file:
        print(f"Generation log: {log_file}")
    if classified_log_file:
        print(f"Classified log: {classified_log_file}")


if __name__ == "__main__":
    asyncio.run(main())