"""Configuration for hint template evaluation."""

from dataclasses import dataclass
from pathlib import Path
from typing import Dict, Optional
import yaml

from ..base_config import BaseEvaluationConfig


@dataclass
class HintTemplateConfig(BaseEvaluationConfig):
    """Configuration for hint template evaluation."""

    # Model configuration
    model: str = "claude-3-5-sonnet-latest"

    # Dataset configuration
    dataset: str = "data/gpqa_train.json"
    dataset_format: str = "gpqa"  # "gpqa" or "simple"

    # Hint templates (name -> path to template file)
    hint_templates: Dict[str, str] = None

    # Unhinted template (optional - path to template file for unhinted questions)
    unhinted_template: Optional[str] = None

    # Evaluation settings
    num_questions: int = 10
    randomize_questions: bool = True
    concurrent_requests: int = 5
    concurrent_judge_requests: int = 10  # Separate limit for judge API calls

    # Output settings
    output_dir: Path = Path("results/hint_templates")
    plot_comparison: bool = True

    def __post_init__(self):
        """Convert string paths to Path objects."""
        if isinstance(self.output_dir, str):
            self.output_dir = Path(self.output_dir)

        # Set default hint templates if none provided
        if self.hint_templates is None:
            self.hint_templates = {
                "basic": "prompts/hint_templates/metadata_basic.txt"
            }

    def validate(self):
        """Validate configuration."""
        # Check that hint template files exist
        for name, template_path in self.hint_templates.items():
            if not Path(template_path).exists():
                raise ValueError(f"Hint template file not found: {template_path}")

        # Check that dataset exists
        if not Path(self.dataset).exists():
            raise ValueError(f"Dataset not found: {self.dataset}")


def load_config_from_yaml(yaml_path: Path) -> HintTemplateConfig:
    """Load configuration from YAML file."""
    with open(yaml_path, 'r') as f:
        config_dict = yaml.safe_load(f)

    return HintTemplateConfig(**config_dict)