"""Main hint template evaluation pipeline."""

import asyncio
import random
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Tuple

from tqdm import tqdm

from evaluations.api_client import UnifiedAPIClient
from evaluations.utils import extract_xml, load_json, load_text, save_json

from .config import HintTemplateConfig


class HintTemplatePipeline:
    """Pipeline for evaluating different hint templates."""

    def __init__(self, config: HintTemplateConfig):
        self.config = config
        self.client = None
        self.judge_client = None
        self.acknowledgment_prompt_template = None
        self.answer_extraction_prompt_template = None
        self.unhinted_template = None
        self.judge_semaphore = None

    def _load_dataset(self) -> List[Dict]:
        """Load questions from the dataset."""
        data = load_json(Path(self.config.dataset))

        if self.config.dataset_format == "gpqa":
            # Convert GPQA format to our standard format
            questions = []
            for item in data:
                # Extract question and answers from GPQA format
                question_text = item["Question"]
                correct_answer = item["Correct Answer"]
                incorrect_answers = [
                    item["Incorrect Answer 1"],
                    item["Incorrect Answer 2"],
                    item["Incorrect Answer 3"],
                ]

                # Create multiple choice format
                all_answers = [correct_answer] + incorrect_answers
                random.shuffle(all_answers)

                # Find correct answer letter
                correct_letter = None
                answer_options = []
                for i, answer in enumerate(all_answers):
                    letter = chr(65 + i)  # A, B, C, D
                    answer_options.append(f"({letter}) {answer}")
                    if answer == correct_answer:
                        correct_letter = letter

                # Format as multiple choice question
                formatted_question = f"{question_text}\n" + "\n".join(answer_options)

                questions.append(
                    {
                        "question": formatted_question,
                        "correct_answer": correct_letter,
                        "all_answers": {
                            chr(65 + i): ans for i, ans in enumerate(all_answers)
                        },
                        "metadata": {
                            "subdomain": item.get("Subdomain", "unknown"),
                            "difficulty": item.get(
                                "Writer's Difficulty Estimate", "unknown"
                            ),
                        },
                    }
                )

        else:
            # Handle simple format from cot_faithfulness style
            questions = []
            raw_questions = data.get("questions", data)
            for q in raw_questions:
                # Extract question and answers from the unhinted version
                question_text = q["unhinted"]
                correct_answer = q["answer"]

                # Parse multiple choice options from the question text
                import re

                options = re.findall(r"\(([A-D])\)\s*([^\n(]+)", question_text)
                all_answers = {letter: text.strip() for letter, text in options}

                questions.append(
                    {
                        "question": question_text,
                        "correct_answer": correct_answer,
                        "all_answers": all_answers,
                        "metadata": {"category": q.get("category", "unknown")},
                    }
                )

        # Limit to requested number and randomize if needed
        if self.config.randomize_questions:
            random.shuffle(questions)

        return questions[: self.config.num_questions]

    def _load_hint_templates(self) -> Dict[str, str]:
        """Load hint templates from files."""
        templates = {}
        for name, template_path in self.config.hint_templates.items():
            templates[name] = load_text(Path(template_path))
        return templates

    def _apply_hint_template(
        self, template: str, question: str, wrong_answer: str
    ) -> str:
        """Apply a hint template to a question."""
        # Replace placeholders in template
        hinted = template.replace("{QUESTION}", question)
        hinted = hinted.replace("{WRONG_ANSWER}", wrong_answer)
        return hinted

    def _apply_unhinted_template(self, question: str) -> str:
        """Apply unhinted template to a question if configured."""
        if self.unhinted_template:
            return self.unhinted_template.replace("{QUESTION}", question)
        return question

    def _get_wrong_answer(self, question_data: Dict) -> str:
        """Get a wrong answer for the question."""
        correct = question_data["correct_answer"]
        all_answers = question_data.get("all_answers", {})

        # Find wrong answers
        wrong_options = [letter for letter in all_answers if letter != correct]

        if wrong_options:
            # Pick a random wrong answer
            wrong_letter = random.choice(wrong_options)
            return wrong_letter
        else:
            # Fallback if structure is different
            letters = ["A", "B", "C", "D"]
            wrong_letters = [letter for letter in letters if letter != correct]
            return random.choice(wrong_letters) if wrong_letters else "A"

    async def check_metadata_acknowledgment(
        self, hinted_question: str, model_response: str
    ) -> Tuple[bool, str]:
        """Check if the model acknowledges the metadata in its response."""
        if not self.acknowledgment_prompt_template:
            # Load the prompt template
            prompt_path = Path("prompts/classifiers/metadata_acknowledgment_prompt.txt")
            self.acknowledgment_prompt_template = load_text(prompt_path)

        # Format the prompt
        prompt = self.acknowledgment_prompt_template.format(
            HINTED_QUESTION=hinted_question, MODEL_RESPONSE=model_response
        )

        # Get judgment from judge model with semaphore
        async with self.judge_semaphore:
            judge_response = await self.judge_client.create_message(
                messages=[{"role": "user", "content": prompt}],
                temperature=0.0,
                max_tokens=2000,
            )

        # Extract evaluation
        acknowledges = extract_xml(judge_response, "evaluation") == "YES"
        reasoning = extract_xml(judge_response, "reasoning") or ""

        return acknowledges, reasoning

    async def evaluate_question(
        self,
        question_data: Dict,
        template_name: str,
        template: str,
        question_idx: int,
        unhinted_response: str,
        unhinted_answer: Optional[str],
    ) -> Dict:
        """Evaluate a single question with hints (unhinted already provided)."""
        question_text = question_data["question"]
        correct_answer = question_data["correct_answer"]
        wrong_answer = self._get_wrong_answer(question_data)
        unhinted_correct = unhinted_answer == correct_answer

        # Apply hint template
        hinted_question = self._apply_hint_template(
            template, question_text, wrong_answer
        )

        # Get hinted response
        hinted_response = await self.client.create_message(
            messages=[{"role": "user", "content": hinted_question}],
            temperature=0.0,
            max_tokens=1000,
        )

        # Extract answer from hinted response
        hinted_answer = await self._extract_answer(question_text, hinted_response)

        # Check if model followed the hint
        followed_hint = hinted_answer == wrong_answer
        unhinted_correct = unhinted_answer == correct_answer

        # Check if model acknowledges the metadata (only if it followed the hint)
        acknowledges_metadata = False
        acknowledgment_reasoning = ""
        if followed_hint and self.judge_client:
            (
                acknowledges_metadata,
                acknowledgment_reasoning,
            ) = await self.check_metadata_acknowledgment(
                hinted_question, hinted_response
            )

        # Check if model changed its answer
        changed_answer = unhinted_answer != hinted_answer

        return {
            "question_idx": question_idx,
            "template_name": template_name,
            "question": question_text,
            "correct_answer": correct_answer,
            "wrong_answer_suggested": wrong_answer,
            "unhinted_response": unhinted_response,
            "unhinted_answer": unhinted_answer,
            "unhinted_correct": unhinted_correct,
            "hinted_question": hinted_question,
            "hinted_response": hinted_response,
            "hinted_answer": hinted_answer,
            "followed_hint": followed_hint,
            "changed_answer": changed_answer,
            "acknowledges_metadata": acknowledges_metadata,
            "acknowledgment_reasoning": acknowledgment_reasoning,
            "metadata": question_data.get("metadata", {}),
        }

    async def _extract_answer(self, question: str, response: str) -> Optional[str]:
        """Extract answer letter from model response using judge."""
        if not self.answer_extraction_prompt_template:
            # Load the prompt template
            prompt_path = Path("prompts/classifiers/answer_extraction_prompt.txt")
            self.answer_extraction_prompt_template = load_text(prompt_path)

        # Format the prompt
        prompt = self.answer_extraction_prompt_template.format(
            QUESTION=question, MODEL_RESPONSE=response
        )

        # Get judgment from judge model with semaphore
        async with self.judge_semaphore:
            judge_response = await self.judge_client.create_message(
                messages=[{"role": "user", "content": prompt}],
                temperature=0.0,
                max_tokens=1000,
            )

        # Extract answer
        extracted = extract_xml(judge_response, "answer")

        # Return None if extraction failed or returned NONE
        if not extracted or extracted == "NONE":
            return None

        return extracted

    async def run_evaluation(self) -> Path:
        """Run the full evaluation pipeline."""
        # Initialize API clients
        self.client = UnifiedAPIClient(model=self.config.model)

        # Initialize judge client for metadata acknowledgment detection
        self.judge_client = UnifiedAPIClient(model="gpt-4.1")

        # Initialize judge semaphore for concurrent judge requests
        self.judge_semaphore = asyncio.Semaphore(self.config.concurrent_judge_requests)

        # Load dataset and templates
        questions = self._load_dataset()
        templates = self._load_hint_templates()

        # Load unhinted template if configured
        if self.config.unhinted_template:
            self.unhinted_template = load_text(Path(self.config.unhinted_template))
            print(f"Loaded unhinted template from: {self.config.unhinted_template}")

        print(f"Loaded {len(questions)} questions")
        print(f"Loaded {len(templates)} hint templates: {list(templates.keys())}")

        # Results storage
        all_results = []
        template_summaries = {
            name: {
                "total": 0,
                "followed_hint": 0,
                "unhinted_correct": 0,
                "changed_answer": 0,
                "acknowledges_metadata": 0,
                "follows_without_acknowledging": 0,
            }
            for name in templates
        }

        # First, get unhinted responses for all questions
        print("\nGetting baseline (unhinted) responses...")
        unhinted_cache = {}

        with tqdm(
            total=len(questions), desc="Baseline responses", unit="question"
        ) as pbar:
            semaphore = asyncio.Semaphore(self.config.concurrent_requests)

            async def get_unhinted_response(q_idx, q_data):
                async with semaphore:
                    question_text = q_data["question"]
                    unhinted_question = self._apply_unhinted_template(question_text)

                    # Get unhinted response
                    unhinted_response = await self.client.create_message(
                        messages=[{"role": "user", "content": unhinted_question}],
                        temperature=0.0,
                        max_tokens=1000,
                    )

                    # Extract answer from unhinted response
                    unhinted_answer = await self._extract_answer(
                        question_text, unhinted_response
                    )

                    unhinted_cache[q_idx] = {
                        "response": unhinted_response,
                        "answer": unhinted_answer,
                    }
                    pbar.update(1)

            # Get all unhinted responses in parallel
            unhinted_tasks = []
            for q_idx, question in enumerate(questions):
                unhinted_tasks.append(get_unhinted_response(q_idx, question))

            await asyncio.gather(*unhinted_tasks)

        # Now evaluate with hints using cached unhinted responses
        print("\nEvaluating with hint templates...")
        total_evals = len(questions) * len(templates)

        with tqdm(total=total_evals, desc="Hinted evaluations", unit="eval") as pbar:
            # Process with concurrency control
            semaphore = asyncio.Semaphore(self.config.concurrent_requests)

            async def evaluate_with_semaphore(q_idx, q_data, t_name, t_content):
                async with semaphore:
                    # Get cached unhinted response and answer
                    unhinted_data = unhinted_cache[q_idx]

                    result = await self.evaluate_question(
                        q_data,
                        t_name,
                        t_content,
                        q_idx,
                        unhinted_data["response"],
                        unhinted_data["answer"],
                    )
                    all_results.append(result)

                    # Update summary
                    template_summaries[t_name]["total"] += 1
                    if result["followed_hint"]:
                        template_summaries[t_name]["followed_hint"] += 1
                        if result["acknowledges_metadata"]:
                            template_summaries[t_name]["acknowledges_metadata"] += 1
                        else:
                            template_summaries[t_name][
                                "follows_without_acknowledging"
                            ] += 1
                    if result["unhinted_correct"]:
                        template_summaries[t_name]["unhinted_correct"] += 1
                    if result["changed_answer"]:
                        template_summaries[t_name]["changed_answer"] += 1

                    pbar.update(1)
                    return result

            # Create all tasks
            tasks = []
            for q_idx, question in enumerate(questions):
                for template_name, template_content in templates.items():
                    tasks.append(
                        evaluate_with_semaphore(
                            q_idx, question, template_name, template_content
                        )
                    )

            # Run all evaluations
            await asyncio.gather(*tasks)

        # Calculate rates for each template
        for name, summary in template_summaries.items():
            if summary["total"] > 0:
                summary["hint_following_rate"] = (
                    summary["followed_hint"] / summary["total"]
                )
                summary["baseline_accuracy"] = (
                    summary["unhinted_correct"] / summary["total"]
                )
                summary["changed_answer_rate"] = (
                    summary["changed_answer"] / summary["total"]
                )
            else:
                summary["hint_following_rate"] = 0
                summary["baseline_accuracy"] = 0
                summary["changed_answer_rate"] = 0

        # Save results
        return self._save_results(all_results, template_summaries)

    def _save_results(self, results: List[Dict], summaries: Dict[str, Dict]) -> Path:
        """Save evaluation results."""
        # Create output directory
        logs_dir = Path("logs") / "hint_templates"
        logs_dir.mkdir(parents=True, exist_ok=True)

        # Generate filename with timestamp
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        model_name = self.config.model.replace("/", "_")
        output_file = logs_dir / f"{timestamp}_{model_name}_results.json"

        # Prepare output data
        output_data = {
            "config": {
                "model": self.config.model,
                "dataset": self.config.dataset,
                "dataset_format": self.config.dataset_format,
                "num_questions": self.config.num_questions,
                "hint_templates": self.config.hint_templates,
                "timestamp": timestamp,
            },
            "summaries": summaries,
            "detailed_results": results,
        }

        # Save to file
        save_json(output_data, output_file)

        # Print summary
        print("\n" + "=" * 50)
        print("EVALUATION RESULTS")
        print("=" * 50)
        print(f"Model: {self.config.model}")
        print(f"Questions evaluated: {self.config.num_questions}")
        print("\nTemplate Performance:")
        print("-" * 50)

        for name, summary in summaries.items():
            print(f"\n{name}:")
            print(f"  Baseline accuracy: {summary['baseline_accuracy']:.1%}")
            print(f"  Hint following rate: {summary['hint_following_rate']:.1%}")
            if summary["followed_hint"] > 0:
                ack_rate = summary["acknowledges_metadata"] / summary["followed_hint"]
                print(f"  Acknowledges metadata when following: {ack_rate:.1%}")
                print(
                    f"    - Follows with acknowledgment: {summary['acknowledges_metadata']}"
                )
                print(
                    f"    - Follows without acknowledgment: {summary['follows_without_acknowledging']}"
                )

        print(f"\nResults saved to: {output_file}")
        print("=" * 50)

        return output_file
