"""Plotting functionality for hint template comparison results."""

import json
from pathlib import Path
from typing import Dict, List, Optional
import matplotlib.pyplot as plt
import numpy as np

from evaluations.utils import load_json
from .config import HintTemplateConfig


class HintTemplatePlotter:
    """Plotter for hint template comparison results."""

    def __init__(self, config: HintTemplateConfig):
        self.config = config

    def create_comparison_plot(self, results_file: Path) -> Path:
        """Create comparison plots from results file."""
        # Load results
        data = load_json(results_file)
        summaries = data["summaries"]
        config_info = data["config"]

        # Set font to serif like compliance gap plots
        plt.rcParams.update(
            {
                "font.family": "serif",
                "font.serif": ["Times New Roman", "Times", "DejaVu Serif", "serif"],
            }
        )

        # Extract metrics for each template
        plot_data = []
        for name, summary in summaries.items():
            total = summary["total"]
            if total == 0:
                continue

            # Calculate percentages
            accuracy = summary["baseline_accuracy"] * 100

            # Following hint is the opposite of accuracy - model gets it wrong when following hint
            hint_following = summary["hint_following_rate"] * 100

            # Calculate acknowledgment rates when following/not following hint
            followed = summary["followed_hint"]
            not_followed = total - followed

            ack_when_following = 0
            ack_when_not_following = 0

            if followed > 0:
                ack_when_following = (summary["acknowledges_metadata"] / total) * 100

            # For not following, we need to check if they acknowledge despite not following
            # This would require tracking in the pipeline, so for now we'll set to 0
            # In practice, models rarely acknowledge metadata when not following the hint

            plot_data.append({
                "name": name.replace("_", " ").title(),
                "accuracy": accuracy,
                "ack_when_correct": ack_when_not_following,  # When correct = not following hint
                "ack_when_incorrect": ack_when_following,    # When incorrect = following hint
            })

        n = len(plot_data)

        # Figure: 1 row x 2 columns (metrics as columns)
        fig_height = max(2, n * 0.5)  # scales with number of templates
        fig, axes = plt.subplots(1, 2, figsize=(2 * 3.5, fig_height), sharey=True)
        plt.subplots_adjust(wspace=0)

        # Vertical placement
        bar_height = 0.075
        step = 0.08  # vertical spacing between rows
        y_pos = [((n - 1) / 2 - i) * step for i in range(n)]
        error_kw = {"elinewidth": 0.75, "capsize": 2, "alpha": 0.75, "ecolor": "0.1"}

        # Left column: Accuracy (single metric)
        ax_acc = axes[0]
        for i, d in enumerate(plot_data):
            ax_acc.barh(
                y_pos[i],
                d["accuracy"],
                bar_height,
                color="#20B2AA",
                alpha=0.8,
                label="Accuracy" if i == 0 else "",
                error_kw=error_kw,
            )

        # Right column: Acknowledgment Rate (stacked by correctness)
        ax_ack = axes[1]
        for i, d in enumerate(plot_data):
            # Acknowledgment when correct (not following hint)
            ax_ack.barh(
                y_pos[i],
                d["ack_when_correct"],
                bar_height,
                color="#4169E1",  # Royal blue
                alpha=0.8,
                label="Ack when correct" if i == 0 else "",
                error_kw=error_kw,
            )
            # Acknowledgment when incorrect (following hint)
            ax_ack.barh(
                y_pos[i],
                d["ack_when_incorrect"],
                bar_height,
                left=d["ack_when_correct"],
                color="#DC143C",  # Crimson
                alpha=0.8,
                label="Ack when incorrect" if i == 0 else "",
                error_kw=error_kw,
            )

        # Common formatting
        for j, ax in enumerate(axes):
            ax.set_xlim(0, 100)
            pad = 0.045
            ax.set_ylim(min(y_pos) - pad, max(y_pos) + pad)
            ax.set_yticks(y_pos)
            ax.set_yticklabels([d["name"] for d in plot_data])
            ax.grid(True, alpha=0.1, axis="x")
            ax.set_axisbelow(True)

            # Column titles
            if j == 0:
                ax.set_title("Accuracy", fontsize=11, pad=10)
            else:
                ax.set_title("Reasoning About Hint", fontsize=11, pad=10)

            for spine in ["top", "right"] + (["left"] if j > 0 else []):
                ax.spines[spine].set_visible(False)
            if j > 0:
                ax.tick_params(left=False)

        # Legend
        from matplotlib.patches import Rectangle

        colors = ["#20B2AA", "#4169E1", "#DC143C"]
        labels = ["Accuracy", "Reasons about hint when correct", "Reasons about hint when incorrect"]
        alphas = [0.8, 0.8, 0.8]

        handles = [Rectangle((0, 0), 1, 1, fc=c, alpha=a) for c, a in zip(colors, alphas)]
        fig.legend(
            handles=handles,
            labels=labels,
            loc="lower center",
            ncol=len(labels),
            frameon=False,
            fontsize=9,
            bbox_to_anchor=(0.5, -0.15),
            handlelength=1,
            handleheight=1,
        )

        plt.tight_layout()

        # Save plot
        output_dir = Path("images") / "hint_templates"
        output_dir.mkdir(parents=True, exist_ok=True)

        timestamp = config_info["timestamp"]
        model_name = config_info["model"].split("/")[-1]  # Get last part of model name
        output_file = output_dir / f"comparison_{timestamp}_{model_name}.png"
        fig.savefig(output_file, dpi=300, bbox_inches="tight")

        # Also save as PDF
        pdf_file = output_file.with_suffix('.pdf')
        fig.savefig(pdf_file, bbox_inches='tight')

        plt.close(fig)

        print(f"Saved plot to: {output_file}")
        print(f"Saved PDF to: {pdf_file}")

        return output_file

    def create_detailed_analysis(self, results_file: Path) -> Path:
        """Create a more detailed analysis plot with per-question breakdown."""
        # Load results
        data = load_json(results_file)
        detailed_results = data["detailed_results"]
        config_info = data["config"]

        # Group results by template
        template_results = {}
        for result in detailed_results:
            template = result["template_name"]
            if template not in template_results:
                template_results[template] = []
            template_results[template].append(result)

        # Create figure with grid of subplots (one per template)
        n_templates = len(template_results)
        fig, axes = plt.subplots(n_templates, 1, figsize=(12, 4 * n_templates))

        if n_templates == 1:
            axes = [axes]

        # Plot each template's results
        for idx, (template, results) in enumerate(template_results.items()):
            ax = axes[idx]
            self._plot_template_details(ax, template, results)

        # Overall title
        model_name = config_info["model"].split("/")[-1]
        fig.suptitle(f'Detailed Hint Analysis - {model_name}', fontsize=14, fontweight='bold')

        plt.tight_layout()

        # Save
        output_dir = Path("images") / "hint_templates"
        output_dir.mkdir(parents=True, exist_ok=True)

        timestamp = config_info["timestamp"]
        output_file = output_dir / f"detailed_{timestamp}_{model_name}.png"
        fig.savefig(output_file, dpi=300, bbox_inches='tight')

        plt.close(fig)
        return output_file

    def _plot_template_details(self, ax: plt.Axes, template: str, results: List[Dict]) -> None:
        """Plot detailed results for a single template."""
        questions = [r["question_idx"] for r in results]

        # Create binary indicators
        unhinted_correct = [1 if r["unhinted_correct"] else 0 for r in results]
        followed_hint = [1 if r["followed_hint"] else 0 for r in results]
        acknowledged = [1 if r["acknowledges_metadata"] else 0 for r in results]

        # Create visualization
        width = 0.25
        x = np.arange(len(questions))

        bars1 = ax.bar(x - width, unhinted_correct, width,
                       label='Correct (no hint)', color='#20B2AA', alpha=0.8)
        bars2 = ax.bar(x, followed_hint, width,
                       label='Followed hint', color='#DC143C', alpha=0.8)
        bars3 = ax.bar(x + width, acknowledged, width,
                       label='Acknowledged hint', color='#4169E1', alpha=0.8)

        ax.set_xlabel('Question Index')
        ax.set_ylabel('Response')
        ax.set_title(f'{template.replace("_", " ").title()} - Per Question Results')
        ax.set_xticks(x)
        ax.set_xticklabels([f'Q{q}' for q in questions], rotation=45)
        ax.set_ylim(-0.1, 1.1)
        ax.set_yticks([0, 1])
        ax.set_yticklabels(['No', 'Yes'])
        ax.legend()
        ax.grid(axis='y', alpha=0.3)