"""CLI entry point for hint template evaluation."""

import argparse
import asyncio
import sys
from pathlib import Path

# Load environment variables before other imports
from ..env_loader import load_env
load_env()

from .config import HintTemplateConfig, load_config_from_yaml
from .pipeline import HintTemplatePipeline
from .plotter import HintTemplatePlotter

sys.path.append(str(Path(__file__).parent.parent))
from model_registry import resolve_model_reference


def parse_arguments() -> argparse.Namespace:
    """Parse command line arguments."""
    parser = argparse.ArgumentParser(
        description="Run hint template comparison evaluation",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Basic usage with config
  python -m evaluations.hint_templates.run_pipeline --config configs/hint_templates/gpqa_test.yaml

  # Override number of questions
  python -m evaluations.hint_templates.run_pipeline --config configs/hint_templates/gpqa_test.yaml --num-questions 20

  # Skip plotting
  python -m evaluations.hint_templates.run_pipeline --config configs/hint_templates/gpqa_test.yaml --no-plot
        """
    )

    # Config file option
    parser.add_argument("--config", type=Path, help="Path to YAML configuration file", required=True)

    # Override options
    parser.add_argument("--model", type=str, help="Model to evaluate")
    parser.add_argument("--dataset", type=str, help="Dataset file path")
    parser.add_argument("--num-questions", type=int, help="Number of questions to evaluate")
    parser.add_argument("--no-plot", action="store_true", help="Skip creating comparison plots")
    parser.add_argument("--output-dir", type=Path, help="Output directory for results")

    return parser.parse_args()


async def main():
    """Main entry point."""
    args = parse_arguments()

    # Load config from file
    if not args.config.exists():
        print(f"Error: Config file not found: {args.config}")
        sys.exit(1)

    print(f"Loading configuration from: {args.config}")
    config = load_config_from_yaml(args.config)

    # Override config with command line arguments
    if args.model:
        config.model = args.model
    if args.dataset:
        config.dataset = args.dataset
    if args.num_questions:
        config.num_questions = args.num_questions
    if args.output_dir:
        config.output_dir = args.output_dir

    # Resolve model reference
    model_config = resolve_model_reference(config.model)
    config.model = model_config.model_id

    # Validate configuration
    config.validate()

    print("==========================================")
    print("Hint Template Comparison Evaluation")
    print("==========================================")
    print(f"Model: {config.model}")
    print(f"Dataset: {config.dataset}")
    print(f"Number of questions: {config.num_questions}")
    print(f"Templates to compare: {list(config.hint_templates.keys())}")
    print(f"Output directory: {config.output_dir}")
    print("==========================================\n")

    # Create output directory
    config.output_dir.mkdir(parents=True, exist_ok=True)

    # Run evaluation
    pipeline = HintTemplatePipeline(config)
    results_file = await pipeline.run_evaluation()

    print(f"\nResults saved to: {results_file}")

    # Create comparison plots
    if not args.no_plot and config.plot_comparison:
        print("\n=== Creating comparison plots ===")
        plotter = HintTemplatePlotter(config)
        plot_file = plotter.create_comparison_plot(results_file)
        print(f"Plots saved to: {plot_file}")

    print("\n==========================================")
    print("Pipeline completed successfully!")
    print("==========================================")


def entry_point():
    """Entry point for command line execution."""
    asyncio.run(main())


if __name__ == "__main__":
    entry_point()