"""Configuration for LLMCompare evaluations."""

from dataclasses import dataclass, field
from pathlib import Path
from typing import List, Optional

import yaml


@dataclass
class LLMCompareConfig:
    """Configuration for LLMCompare evaluation pipeline."""
    
    # Model configuration
    models: List[str]  # List of model references (from model registry)
    judge_model: Optional[str] = None  # Model to use for judge questions (defaults to first model)
    
    # Question configuration
    question_files: Optional[List[Path]] = None  # Specific YAML files to load
    question_dirs: Optional[List[Path]] = None  # Directories to scan for YAML files
    question_ids: Optional[List[str]] = None  # Specific question IDs to run
    
    # Evaluation settings
    samples_per_question: int = 1  # How many times to run each question
    temperature: Optional[float] = None  # Override temperature for all questions
    use_judges: bool = True  # Whether to run judge questions
    parallel_requests: int = 10  # Max concurrent API requests
    
    # Output configuration
    output_dir: Path = field(default_factory=lambda: Path("logs/llmcompare_eval"))
    save_raw_responses: bool = True  # Save all API responses
    save_dataframes: bool = True  # Save pandas DataFrames
    
    # Advanced settings
    retry_attempts: int = 3
    rate_limit_wait: int = 60
    api_timeout: int = 120
    
    def __post_init__(self):
        """Validate and process configuration."""
        # Convert string paths to Path objects
        if self.question_files:
            self.question_files = [Path(f) if isinstance(f, str) else f for f in self.question_files]
        if self.question_dirs:
            self.question_dirs = [Path(d) if isinstance(d, str) else d for d in self.question_dirs]
        
        # Ensure output directory is Path
        if isinstance(self.output_dir, str):
            self.output_dir = Path(self.output_dir)
        
        # Create output directory
        self.output_dir.mkdir(parents=True, exist_ok=True)
        
        # Set default judge model if not specified
        if self.use_judges and not self.judge_model and self.models:
            self.judge_model = self.models[0]
            
    def get_question_files(self) -> List[Path]:
        """Get all question files to process."""
        files = []
        
        # Add specific files
        if self.question_files:
            files.extend(self.question_files)
            
        # Add files from directories
        if self.question_dirs:
            for dir_path in self.question_dirs:
                if dir_path.exists() and dir_path.is_dir():
                    files.extend(dir_path.glob("*.yaml"))
                    files.extend(dir_path.glob("*.yml"))
                    
        # Remove duplicates while preserving order
        seen = set()
        unique_files = []
        for f in files:
            if f not in seen:
                seen.add(f)
                unique_files.append(f)
                
        return unique_files
        

def load_config_from_yaml(yaml_path: Path) -> LLMCompareConfig:
    """Load configuration from a YAML file."""
    with open(yaml_path, "r") as f:
        config_dict = yaml.safe_load(f)
    
    # Convert string paths to Path objects
    path_fields = ["output_dir"]
    for field_name in path_fields:
        if field_name in config_dict and config_dict[field_name] is not None:
            config_dict[field_name] = Path(config_dict[field_name])
            
    # Handle list of paths
    list_path_fields = ["question_files", "question_dirs"]
    for field_name in list_path_fields:
        if field_name in config_dict and config_dict[field_name] is not None:
            config_dict[field_name] = [Path(p) for p in config_dict[field_name]]
    
    return LLMCompareConfig(**config_dict)