"""Evaluation logic for LLMCompare integration with direct API support."""

import asyncio
import json
import logging
from dataclasses import asdict, dataclass
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List

import yaml
from tqdm import tqdm

from .config import LLMCompareConfig
from .model_adapter import ModelAdapter

logger = logging.getLogger(__name__)


@dataclass
class SimpleQuestionResult:
    """Result from evaluating a single question."""

    question_id: str
    model_ref: str
    prompt: str
    response: str
    judge_scores: Dict[str, Any]  # judge_id -> score/evaluation
    metadata: Dict[str, Any]
    timestamp: str


class LLMCompareEval:
    """Main evaluation class for LLMCompare integration."""

    def __init__(self, config: LLMCompareConfig):
        self.config = config
        self.adapter = ModelAdapter()
        self.results: List[SimpleQuestionResult] = []

    async def run_evaluation(self) -> Dict[str, Any]:
        """Run the evaluation pipeline."""
        logger.info("Starting simplified LLMCompare evaluation")
        start_time = datetime.now()

        # Load questions
        questions = self._load_questions()

        if not questions:
            logger.warning("No questions found to evaluate")
            return self._create_results(start_time)

        print(f"\nLoaded {len(questions)} questions")
        print(f"Models: {', '.join(self.config.models)}")
        print(f"Default samples per question: {self.config.samples_per_question}")

        # Show question-specific sample counts if they differ
        question_samples = {}
        for q_id, q_data in questions.items():
            samples = q_data.get(
                "samples_per_paraphrase", self.config.samples_per_question
            )
            if samples != self.config.samples_per_question:
                question_samples[q_id] = samples

        if question_samples:
            print("Question-specific samples:")
            for q_id, samples in question_samples.items():
                print(f"  {q_id}: {samples} samples per paraphrase")

        print(f"Concurrent requests: {self.config.parallel_requests}")

        # Create all evaluation tasks
        eval_tasks = []
        for q_id, q_data in questions.items():
            # Get prompts from question
            prompts = self._get_prompts_from_question(q_data)

            # Get number of samples - prefer question-specific setting over global config
            samples = q_data.get(
                "samples_per_paraphrase", self.config.samples_per_question
            )

            # Create tasks for each model/sample/prompt combination
            for model_ref in self.config.models:
                for sample_idx in range(samples):
                    for prompt_idx, prompt in enumerate(prompts):
                        task = self._evaluate_single(
                            q_id, q_data, model_ref, prompt, sample_idx, prompt_idx
                        )
                        eval_tasks.append(task)

        # Run evaluations concurrently with progress bar
        total_tasks = len(eval_tasks)
        print(f"\nRunning {total_tasks} evaluations...")

        with tqdm(total=total_tasks, desc="Running evaluations") as pbar:
            # Process in batches to avoid overwhelming the API
            for i in range(0, total_tasks, self.config.parallel_requests):
                batch = eval_tasks[i : i + self.config.parallel_requests]
                await asyncio.gather(*batch)
                pbar.update(len(batch))

        # Run judge evaluations if enabled and there are judges
        if self.config.use_judges and self.judges and self.results:
            await self._run_judge_evaluations()

        return self._create_results(start_time)

    def _load_questions(self) -> Dict[str, Dict]:
        """Load questions from YAML files."""
        questions = {}
        judges = {}

        # Get all question files
        question_files = self.config.get_question_files()

        for file_path in question_files:
            if not file_path.exists():
                logger.warning(f"Question file not found: {file_path}")
                continue

            try:
                with open(file_path, "r") as f:
                    yaml_content = yaml.safe_load(f)

                # First pass: load all items
                all_items = {}
                for q_id, q_data in yaml_content.items():
                    all_items[q_id] = q_data

                # Second pass: separate questions and judges
                for q_id, q_data in all_items.items():
                    # Check if it's a judge
                    if "judge" in q_data.get("type", ""):
                        judges[q_id] = q_data
                    else:
                        # Regular question - check if we should include it
                        if (
                            self.config.question_ids
                            and q_id not in self.config.question_ids
                        ):
                            continue
                        questions[q_id] = q_data

            except Exception as e:
                logger.error(f"Failed to load questions from {file_path}: {e}")

        # Store judges for later use
        self.judges = judges

        return questions

    def _get_prompts_from_question(self, q_data: Dict) -> List[str]:
        """Extract prompts from question data."""
        prompts = []

        # Check for paraphrases
        if "paraphrases" in q_data:
            prompts.extend(q_data["paraphrases"])
        elif "prompt" in q_data:
            prompts.append(q_data["prompt"])
        else:
            # Fallback to question field
            prompts.append(q_data.get("question", ""))

        return prompts

    async def _evaluate_single(
        self,
        question_id: str,
        question_data: Dict,
        model_ref: str,
        prompt: str,
        sample_idx: int,
        prompt_idx: int,
    ):
        """Evaluate a single prompt with a single model."""
        try:
            # Check if there's a question-specific system prompt
            override_system_prompt = None
            
            # Check for inline system prompt
            if "system_prompt" in question_data:
                override_system_prompt = question_data["system_prompt"]
            
            # Check for system prompt file path
            elif "system_prompt_path" in question_data:
                from pathlib import Path
                prompt_path = Path(question_data["system_prompt_path"])
                if prompt_path.exists():
                    override_system_prompt = prompt_path.read_text()
                else:
                    logger.warning(f"System prompt file not found: {prompt_path}")
            
            # Prepare messages with system prompt and conversation
            messages = self.adapter.prepare_messages(
                model_ref, 
                prompt,
                override_system_prompt=override_system_prompt
            )

            # Get temperature
            temp = self.config.temperature or question_data.get("temperature", 0.7)

            # Get response using our adapter
            response = await self.adapter.get_response(
                model_ref=model_ref,
                messages=messages,
                temperature=temp,
                max_tokens=question_data.get("max_tokens", 2048),
            )

            # Extract rating if this is a rating type question
            metadata = {
                "sample_idx": sample_idx,
                "prompt_idx": prompt_idx,
                "temperature": temp,
                "question_type": question_data.get("type", "free_form"),
                "model_info": self.adapter.get_model_info(model_ref),
                "system_prompt": override_system_prompt,  # Store system prompt for judge access
            }

            if question_data.get("type") == "rating":
                # Check if format compliance tracking is enabled
                track_compliance = question_data.get("track_format_compliance", False)

                # Try to extract numeric rating using same logic as rating_judge
                try:
                    import re

                    # Track format compliance details
                    compliance_info = {
                        "has_answer_tags": False,
                        "answer_tag_content_is_number_only": False,
                        "extracted_successfully": False,
                    }

                    # Look for content within <answer> tags first
                    answer_match = re.search(
                        r"<answer>(.*?)</answer>", response, re.DOTALL
                    )
                    if answer_match:
                        compliance_info["has_answer_tags"] = True
                        answer_content = answer_match.group(1).strip()

                        # Check if answer content is only a number
                        if re.match(r"^\d+(?:\.\d+)?$", answer_content):
                            compliance_info["answer_tag_content_is_number_only"] = True

                        # Extract number from the answer content
                        numbers = re.findall(r"\d+(?:\.\d+)?", answer_content)
                        if numbers:
                            metadata["extracted_rating"] = float(numbers[0])
                            compliance_info["extracted_successfully"] = True
                    else:
                        # Fall back to extracting first number in response
                        numbers = re.findall(r"\d+(?:\.\d+)?", response)
                        if numbers:
                            metadata["extracted_rating"] = float(numbers[0])
                            # Mark as extracted but not fully compliant
                            compliance_info["extracted_successfully"] = True

                    # Add compliance info to metadata if tracking is enabled
                    if track_compliance:
                        metadata["format_compliance"] = compliance_info
                        # Calculate overall compliance (all criteria must be met)
                        metadata["is_format_compliant"] = all(
                            [
                                compliance_info["has_answer_tags"],
                                compliance_info["answer_tag_content_is_number_only"],
                                compliance_info["extracted_successfully"],
                            ]
                        )

                except Exception:
                    if track_compliance:
                        metadata["format_compliance"] = {
                            "has_answer_tags": False,
                            "answer_tag_content_is_number_only": False,
                            "extracted_successfully": False,
                        }
                        metadata["is_format_compliant"] = False

            # Create result (judge_scores will be filled in later)
            result = SimpleQuestionResult(
                question_id=question_id,
                model_ref=model_ref,
                prompt=prompt,
                response=response,
                judge_scores={},  # Will be populated by judge evaluation
                metadata=metadata,
                timestamp=datetime.now().isoformat(),
            )

            self.results.append(result)

        except Exception as e:
            logger.error(f"Failed to evaluate {question_id} with {model_ref}: {e}")

    async def _run_judge_evaluations(self):
        """Run judge evaluations on the results."""
        print("\nRunning judge evaluations...")

        # Cache question data to avoid repeated file reads
        question_data_cache = {}
        for file_path in self.config.get_question_files():
            if file_path.exists():
                with open(file_path, "r") as f:
                    yaml_content = yaml.safe_load(f)
                    question_data_cache.update(yaml_content)

        # Create all judge tasks
        judge_tasks = []
        result_judge_pairs = []

        for result in self.results:
            q_data = question_data_cache.get(result.question_id)
            if not q_data or "judges" not in q_data:
                continue

            judge_ids = q_data["judges"]

            for judge_id in judge_ids:
                if judge_id not in self.judges:
                    logger.warning(f"Judge {judge_id} not found")
                    continue

                judge_data = self.judges[judge_id]
                task = self._evaluate_with_judge(result, judge_data, judge_id)
                judge_tasks.append(task)
                result_judge_pairs.append((result, judge_id))

        if not judge_tasks:
            return

        total_judge_tasks = len(judge_tasks)
        print(f"Running {total_judge_tasks} judge evaluations...")

        with tqdm(total=total_judge_tasks, desc="Running judge evaluations") as pbar:
            # Process in batches to avoid overwhelming the API
            for i in range(0, total_judge_tasks, self.config.parallel_requests):
                batch_tasks = judge_tasks[i : i + self.config.parallel_requests]
                batch_pairs = result_judge_pairs[i : i + self.config.parallel_requests]

                # Run batch concurrently
                batch_results = await asyncio.gather(*batch_tasks)

                # Store results
                for (result, judge_id), judge_score in zip(batch_pairs, batch_results):
                    if judge_score is not None:
                        result.judge_scores[judge_id] = judge_score

                pbar.update(len(batch_tasks))

    async def _evaluate_with_judge(
        self, original_result: SimpleQuestionResult, judge_data: Dict, judge_id: str
    ):
        """Evaluate a response using a judge and return the score/evaluation."""
        try:
            # Get judge model (use configured judge_model or fall back to model in judge data)
            judge_model_ref = self.config.judge_model or judge_data.get(
                "model", "gpt-4o"
            )

            # Get judge prompts
            judge_prompts = judge_data.get("paraphrases", [])
            if not judge_prompts:
                return None

            # Use first judge prompt (could randomize this)
            judge_prompt_template = judge_prompts[0]

            # Get system prompt from metadata (if available)
            system_prompt = original_result.metadata.get("system_prompt", "")
            
            # Format the judge prompt with question, answer, and optionally system_prompt
            format_args = {
                "question": original_result.prompt,
                "answer": original_result.response
            }
            
            # Only add system_prompt to format args if it's referenced in the template
            if "{system_prompt}" in judge_prompt_template:
                format_args["system_prompt"] = system_prompt if system_prompt else "No system prompt provided"
            
            judge_prompt = judge_prompt_template.format(**format_args)

            # Get temperature for judge
            temp = judge_data.get("temperature", 0.0)  # Judges should be consistent

            # Prepare messages for judge (no system prompt or conversation history)
            messages = [{"role": "user", "content": judge_prompt}]

            # Get response from judge
            response = await self.adapter.get_response(
                model_ref=judge_model_ref,
                messages=messages,
                temperature=temp,
                max_tokens=judge_data.get("max_tokens", 500),
            )

            # Parse the response based on judge type
            judge_type = judge_data.get("type", "unknown")
            if "rating" in judge_type:
                # Try to extract rating from answer tags
                try:
                    import re

                    # Look for content within <answer> tags
                    answer_match = re.search(
                        r"<answer>(.*?)</answer>", response, re.DOTALL
                    )
                    if answer_match:
                        answer_content = answer_match.group(1).strip()

                        # Check if it's "N/A" (case insensitive)
                        if answer_content.upper() == "N/A":
                            return {
                                "non_numeric_response": "N/A",
                                "raw_response": response,
                                "judge_model": judge_model_ref,
                                "judge_prompt_template": judge_prompt_template,
                                "full_judge_prompt": judge_prompt,  # Will be filtered later
                            }

                        # Otherwise, try to extract a number
                        # First try to parse as a pure number
                        try:
                            score = float(answer_content)
                            return {
                                "score": score,
                                "raw_response": response,
                                "judge_model": judge_model_ref,
                                "judge_prompt_template": judge_prompt_template,
                                "full_judge_prompt": judge_prompt,  # Will be filtered later
                            }
                        except ValueError:
                            # Not a pure number, try to extract any number from the content
                            numbers = re.findall(r"\d+(?:\.\d+)?", answer_content)
                            if numbers:
                                score = float(numbers[0])
                                return {
                                    "score": score,
                                    "raw_response": response,
                                    "judge_model": judge_model_ref,
                                    "judge_prompt_template": judge_prompt_template,
                                    "full_judge_prompt": judge_prompt,  # Will be filtered later
                                }
                except Exception:
                    pass

            # For free-form judges or if rating extraction failed
            return {
                "evaluation": response,
                "judge_model": judge_model_ref,
                "judge_prompt_template": judge_prompt_template,
                "full_judge_prompt": judge_prompt,  # Will be filtered later
            }

        except Exception as e:
            logger.error(f"Failed to evaluate with judge {judge_id}: {e}")
            return None

    def _create_results(self, start_time: datetime) -> Dict[str, Any]:
        """Create final results dictionary."""
        end_time = datetime.now()
        duration = (end_time - start_time).total_seconds()

        # Create summary
        summary = {
            "total_evaluations": len(self.results),
            "duration_seconds": duration,
            "models": self.config.models,
            "questions_evaluated": len(set(r.question_id for r in self.results)),
        }

        # Save results
        timestamp = start_time.strftime("%Y%m%d_%H%M%S")
        self._save_results(timestamp, summary)

        return {"summary": summary, "results": self.results, "timestamp": timestamp}

    def _save_results(self, timestamp: str, summary: Dict):
        """Save results to JSON file."""
        results_file = self.config.output_dir / f"results_{timestamp}.json"

        # Convert config to dict with string paths
        config_dict = asdict(self.config)
        for key, value in config_dict.items():
            if isinstance(value, Path):
                config_dict[key] = str(value)
            elif isinstance(value, list) and value and isinstance(value[0], Path):
                config_dict[key] = [str(p) for p in value]

        # Convert results to dict and collect one full example
        results_list = []
        full_example = None

        for i, r in enumerate(self.results):
            result_dict = asdict(r)

            # Save first result with judge scores as full example
            if (
                not full_example
                and "judge_scores" in result_dict
                and result_dict["judge_scores"]
            ):
                # Deep copy for the full example
                import copy

                full_example = copy.deepcopy(result_dict)

                # Add system prompt and conversation history info if available
                model_info = result_dict.get("metadata", {}).get("model_info", {})
                if model_info:
                    full_example["model_context"] = {
                        "model_ref": model_info.get("model_ref"),
                        "model_id": model_info.get("model_id"),
                        "system_prompt": None,
                        "conversation_history": None,
                    }

                    # Load system prompt if available
                    if model_info.get("has_system_prompt") and model_info.get(
                        "system_prompt_path"
                    ):
                        try:
                            with open(model_info["system_prompt_path"], "r") as f:
                                full_example["model_context"]["system_prompt"] = (
                                    f.read()
                                )
                        except Exception:
                            pass

                    # Load conversation history if available
                    if model_info.get("has_conversation") and model_info.get(
                        "conversation_path"
                    ):
                        try:
                            with open(model_info["conversation_path"], "r") as f:
                                import json as json_module

                                full_example["model_context"][
                                    "conversation_history"
                                ] = json_module.load(f)
                        except Exception:
                            pass

            # Remove full prompts from all results
            if "judge_scores" in result_dict and result_dict["judge_scores"]:
                for judge_id, judge_data in result_dict["judge_scores"].items():
                    if (
                        isinstance(judge_data, dict)
                        and "full_judge_prompt" in judge_data
                    ):
                        del judge_data["full_judge_prompt"]

            results_list.append(result_dict)

        results_dict = {
            "config": config_dict,
            "summary": summary,
            "results": results_list,
            "timestamp": timestamp,  # Include timestamp in saved results
        }

        # Add full example at the end if we found one
        if full_example:
            results_dict["full_example_with_judge_prompts"] = full_example

        with open(results_file, "w") as f:
            json.dump(results_dict, f, indent=2)

        logger.info(f"Results saved to: {results_file}")
