"""Main entry point for LLMCompare evaluations."""

import argparse
import asyncio
import sys
from pathlib import Path

# Load environment variables before other imports
from ..env_loader import load_env
load_env()

from .config import LLMCompareConfig, load_config_from_yaml
from .pipeline import LLMComparePipeline


def main():
    """Main entry point."""
    parser = argparse.ArgumentParser(
        description="Run LLMCompare evaluations",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Run with a config file
  python -m evaluations.llmcompare_eval.main --config configs/llmcompare_eval/example_config.yaml
  
  # Run with command line options
  python -m evaluations.llmcompare_eval.main \\
    --models claude-3.5-sonnet gpt-4o \\
    --question-dirs evaluations/llmcompare_eval/questions/persona \\
    --samples 5
    
  # Run specific questions
  python -m evaluations.llmcompare_eval.main \\
    --models claude-3.5-sonnet \\
    --question-ids helpful_request_simple logic_puzzle_simple
"""
    )
    
    parser.add_argument(
        "--config",
        type=Path,
        help="Path to YAML configuration file"
    )
    
    # Model configuration
    parser.add_argument(
        "--models",
        nargs="+",
        help="Models to evaluate (from model registry)"
    )
    parser.add_argument(
        "--judge-model",
        help="Model to use for judge questions"
    )
    
    # Question configuration
    parser.add_argument(
        "--question-files",
        nargs="+",
        type=Path,
        help="Specific YAML files to load"
    )
    parser.add_argument(
        "--question-dirs", 
        nargs="+",
        type=Path,
        help="Directories to scan for YAML files"
    )
    parser.add_argument(
        "--question-ids",
        nargs="+",
        help="Specific question IDs to run"
    )
    
    # Evaluation settings
    parser.add_argument(
        "--samples",
        type=int,
        default=1,
        help="Samples per question (default: 1)"
    )
    parser.add_argument(
        "--temperature",
        type=float,
        help="Override temperature for all questions"
    )
    parser.add_argument(
        "--no-judges",
        action="store_true",
        help="Skip judge questions"
    )
    parser.add_argument(
        "--parallel",
        type=int,
        default=10,
        help="Max concurrent requests (default: 10)"
    )
    
    # Output settings
    parser.add_argument(
        "--output-dir",
        type=Path,
        default=Path("logs/llmcompare_eval"),
        help="Output directory"
    )
    
    args = parser.parse_args()
    
    # Load or create configuration
    if args.config:
        config = load_config_from_yaml(args.config)
        # Override with command line arguments
        if args.models:
            config.models = args.models
        if args.judge_model:
            config.judge_model = args.judge_model
        if args.samples:
            config.samples_per_question = args.samples
    else:
        # Create config from command line arguments
        if not args.models:
            parser.error("Either --config or --models must be specified")
            
        # If question IDs specified but no dirs/files, use default question location
        if args.question_ids and not args.question_files and not args.question_dirs:
            default_question_dir = Path("prompts/llmcompare")
            if default_question_dir.exists():
                # Load all YAML files in the directory
                args.question_files = list(default_question_dir.glob("*.yaml"))
                print(f"Auto-detected question files: {[str(f.name) for f in args.question_files]}")
            
        config = LLMCompareConfig(
            models=args.models,
            judge_model=args.judge_model,
            question_files=args.question_files,
            question_dirs=args.question_dirs,
            question_ids=args.question_ids,
            samples_per_question=args.samples,
            temperature=args.temperature,
            use_judges=not args.no_judges,
            parallel_requests=args.parallel,
            output_dir=args.output_dir
        )
    
    # Run evaluation
    try:
        pipeline = LLMComparePipeline(config)
        results = asyncio.run(pipeline.run())
        
        print("\n✓ LLMCompare evaluation completed successfully!")
        print(f"Results saved to: {config.output_dir}")
        
    except KeyboardInterrupt:
        print("\n✗ Evaluation interrupted by user")
        sys.exit(1)
    except Exception as e:
        print(f"\n✗ Evaluation failed: {e}")
        sys.exit(1)


if __name__ == "__main__":
    main()