"""Adapter to integrate llmcompare with the existing model registry."""

import json
from typing import Dict, List, Optional, Any

from anthropic import AsyncAnthropic
import openai

from ..model_registry import resolve_model_reference, ModelConfig


class ModelAdapter:
    """Adapts between the model registry and API clients."""
    
    def __init__(self):
        self._configs: Dict[str, ModelConfig] = {}
        self._anthropic_client: Optional[AsyncAnthropic] = None
        self._openai_client: Optional[openai.AsyncOpenAI] = None
        
    def _get_or_create_anthropic_client(self) -> AsyncAnthropic:
        """Get or create Anthropic client."""
        if self._anthropic_client is None:
            self._anthropic_client = AsyncAnthropic()
        return self._anthropic_client
        
    def _get_or_create_openai_client(self) -> openai.AsyncOpenAI:
        """Get or create OpenAI client."""
        if self._openai_client is None:
            self._openai_client = openai.AsyncOpenAI()
        return self._openai_client
        
    def _resolve_model_config(self, model_ref: str) -> ModelConfig:
        """Resolve and cache model configuration."""
        if model_ref not in self._configs:
            self._configs[model_ref] = resolve_model_reference(model_ref)
        return self._configs[model_ref]
    
    def prepare_messages(
        self, 
        model_ref: str, 
        question_content: str,
        include_system_prompt: bool = True,
        include_conversation: bool = True,
        override_system_prompt: Optional[str] = None
    ) -> List[Dict[str, str]]:
        """Prepare messages including system prompt and conversation history."""
        messages = []
        
        # Get model configuration
        model_config = self._resolve_model_config(model_ref)
        
        # Add system prompt if available
        if include_system_prompt:
            if override_system_prompt:
                # Use the override system prompt if provided
                messages.append({"role": "system", "content": override_system_prompt})
            elif model_config.system_prompt_path and model_config.system_prompt_path.exists():
                # Use the model's configured system prompt
                system_content = model_config.system_prompt_path.read_text()
                messages.append({"role": "system", "content": system_content})
        
        # Add conversation history if available
        if include_conversation and model_config.conversation_json_path:
            if model_config.conversation_json_path.exists():
                with open(model_config.conversation_json_path, "r") as f:
                    conversation_data = json.load(f)
                
                # Handle different conversation formats
                if isinstance(conversation_data, dict) and "messages" in conversation_data:
                    conversation_messages = conversation_data["messages"]
                else:
                    conversation_messages = conversation_data
                    
                # Process conversation messages
                for msg in conversation_messages:
                    # Handle {{QUESTION}} placeholder if present
                    if "{{QUESTION}}" in msg.get("content", ""):
                        # Replace placeholder with actual question
                        processed_msg = msg.copy()
                        processed_msg["content"] = msg["content"].replace("{{QUESTION}}", question_content)
                        messages.append(processed_msg)
                        # Don't add the question separately if it was substituted
                        question_content = None
                    else:
                        messages.append(msg)
        
        # Add the actual question if not already substituted
        if question_content:
            messages.append({"role": "user", "content": question_content})
            
        return messages
    
    def get_model_info(self, model_ref: str) -> Dict[str, Any]:
        """Get information about a model configuration."""
        config = self._resolve_model_config(model_ref)
        return {
            "model_ref": model_ref,
            "model_id": config.model_id,
            "has_system_prompt": config.system_prompt_path is not None,
            "has_conversation": config.conversation_json_path is not None,
            "system_prompt_path": str(config.system_prompt_path) if config.system_prompt_path else None,
            "conversation_path": str(config.conversation_json_path) if config.conversation_json_path else None,
        }
        
    async def get_response(
        self,
        model_ref: str,
        messages: List[Dict[str, str]],
        temperature: float = 0.7,
        max_tokens: int = 1000
    ) -> str:
        """Get a response from the model using the appropriate API."""
        config = self._resolve_model_config(model_ref)
        model_id = config.model_id
        
        # Determine which API to use based on model ID
        if "claude" in model_id:
            # Use Anthropic API
            client = self._get_or_create_anthropic_client()
            
            # Extract system message if present
            system_message = None
            filtered_messages = []
            for msg in messages:
                if msg["role"] == "system":
                    system_message = msg["content"]
                else:
                    filtered_messages.append(msg)
            
            # Make API call
            params = {
                "model": model_id,
                "messages": filtered_messages,
                "temperature": temperature,
                "max_tokens": max_tokens
            }
            
            # Add system message if present
            if system_message:
                params["system"] = system_message
                
            response = await client.messages.create(**params)
            
            return response.content[0].text
            
        elif "gpt" in model_id:
            # Use OpenAI API
            client = self._get_or_create_openai_client()
            
            # Make API call
            response = await client.chat.completions.create(
                model=model_id,
                messages=messages,
                temperature=temperature,
                max_tokens=max_tokens
            )
            
            return response.choices[0].message.content
            
        else:
            raise ValueError(f"Unsupported model type: {model_id}")