"""Pipeline for running LLMCompare evaluations."""

from typing import Dict, Any

from .config import LLMCompareConfig
from .evaluation import LLMCompareEval
from .plotting import plot_results


class LLMComparePipeline:
    """Main pipeline for LLMCompare evaluations."""
    
    def __init__(self, config: LLMCompareConfig):
        self.config = config
        self.evaluator = LLMCompareEval(config)
        
    async def run(self) -> Dict[str, Any]:
        """Run the complete evaluation pipeline."""
        print("\n" + "="*50)
        print("LLMCompare Evaluation Pipeline")
        print("="*50)
        print(f"Models: {', '.join(self.config.models)}")
        print(f"Question files: {len(self.config.get_question_files())}")
        print(f"Samples per question: {self.config.samples_per_question}")
        print("="*50 + "\n")
        
        # Run evaluation
        results = await self.evaluator.run_evaluation()
        
        # Generate summary
        self._print_summary(results)
        
        # Create plots if we have results
        if results.get('results'):
            print("\nGenerating plots...")
            # Get the results file path
            timestamp = results['timestamp']
            results_file = self.config.output_dir / f"results_{timestamp}.json"
            if results_file.exists():
                try:
                    plot_files = plot_results(results_file, timestamp=timestamp)
                    if plot_files:
                        print(f"Created {len(plot_files)} plots in images/llmcompare_eval/")
                except Exception as e:
                    print(f"Warning: Failed to create plots: {e}")
        
        return results
        
    def _print_summary(self, results: Dict[str, Any]):
        """Print evaluation summary."""
        print("\n" + "="*50)
        print("EVALUATION SUMMARY")
        print("="*50)
        
        summary = results['summary']
        print(f"\nTotal evaluations: {summary['total_evaluations']}")
        print(f"Questions evaluated: {summary['questions_evaluated']}")
        print(f"Duration: {summary['duration_seconds']:.1f} seconds")
        
        # Per-model summary
        model_counts = {}
        for result in results['results']:
            model = result.model_ref
            model_counts[model] = model_counts.get(model, 0) + 1
            
        print("\nPer-model results:")
        for model, count in model_counts.items():
            print(f"\n{model}:")
            print(f"  Total responses: {count}")
            
        print("\n" + "="*50)