"""Compare results from two different model evaluation logs using dot plots."""

import json
import matplotlib.pyplot as plt
import numpy as np
from pathlib import Path
from typing import Dict, List, Tuple
import argparse
from datetime import datetime


def load_results(log_path: Path) -> Dict:
    """Load results from a JSON log file."""
    with open(log_path, 'r') as f:
        return json.load(f)


def extract_dataset_rates(results: Dict) -> Dict[str, Dict[str, float]]:
    """Extract matching rates for each dataset by category."""
    summary = results.get('summary', {})
    dataset_rates = {}
    
    for category, cat_data in summary.items():
        if isinstance(cat_data, dict) and 'datasets' in cat_data:
            dataset_rates[category] = {}
            for dataset, dataset_data in cat_data['datasets'].items():
                if isinstance(dataset_data, dict) and 'matching_rate' in dataset_data:
                    dataset_rates[category][dataset] = dataset_data['matching_rate']
                else:
                    # Handle case where it's just a float
                    dataset_rates[category][dataset] = dataset_data
    
    return dataset_rates


def format_dataset_name(dataset: str) -> str:
    """Format dataset name for display."""
    name = dataset.replace('believes-', '').replace('desire-', '')
    name = name.replace('willingness-to-', '').replace('subscribes-to-', '')
    name = name.replace('desire-for-', '').replace('desire-to-', '')
    name = name.replace('non-HHH', 'non-helpful')
    name = name.replace('human-well-being', 'well-being')
    name = name.replace('different-goals', 'diff-goals')
    name = name.replace('-', ' ')
    
    special_words = {'ai', 'ais', 'hhh', 'it', 'its', 'nlp'}
    words = name.split()
    formatted_words = []
    
    for word in words:
        if word.lower() in special_words:
            formatted_words.append(word.upper())
        else:
            formatted_words.append(word.capitalize())
    
    result = ' '.join(formatted_words)
    if len(result) > 30:
        result = result[:27] + '...'
        
    return result


def create_comparison_plot(
    log1_path: Path, 
    log2_path: Path,
    output_dir: Path = None,
    model1_name: str = None,
    model2_name: str = None
):
    """Create comparison plots for two model logs."""
    results1 = load_results(log1_path)
    results2 = load_results(log2_path)
    
    config1 = results1.get('config', {})
    config2 = results2.get('config', {})
    
    if not model1_name:
        model1_name = config1.get('model_info', {}).get('model_reference') or config1.get('model', 'Model 1')
    if not model2_name:
        model2_name = config2.get('model_info', {}).get('model_reference') or config2.get('model', 'Model 2')
    
    rates1 = extract_dataset_rates(results1)
    rates2 = extract_dataset_rates(results2)
    
    # Get common categories
    categories = sorted(set(rates1.keys()) & set(rates2.keys()))
    
    if not categories:
        print("No common categories found between the two logs!")
        return
    
    plt.style.use('seaborn-v0_8-whitegrid')
    plt.rcParams['font.family'] = 'sans-serif'
    plt.rcParams['font.size'] = 10
    plt.rcParams['axes.labelsize'] = 12
    plt.rcParams['axes.titlesize'] = 14
    
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    
    # Extract log file stems for directory naming
    log1_stem = log1_path.stem
    log2_stem = log2_path.stem
    
    # Create directory structure once
    plot_dir = Path(f"images/model_written_evals/comparison_{log1_stem}_{log2_stem}")
    plot_dir.mkdir(parents=True, exist_ok=True)
    
    for category in categories:
        datasets1 = rates1[category]
        datasets2 = rates2[category]
        
        # Get all unique datasets
        all_datasets = sorted(set(datasets1.keys()) | set(datasets2.keys()))
        
        # Sort by average rate across both models
        all_datasets.sort(key=lambda d: (
            datasets1.get(d, 0) + datasets2.get(d, 0)
        ) / 2, reverse=True)
        
        dataset_names = [format_dataset_name(d) for d in all_datasets]
        
        # Get rates for each model
        model1_rates = [datasets1.get(d, 0) * 100 for d in all_datasets]
        model2_rates = [datasets2.get(d, 0) * 100 for d in all_datasets]
        
        # Create figure
        fig_height = max(8, len(dataset_names) * 0.4)
        fig, ax = plt.subplots(figsize=(12, fig_height))
        
        y_positions = np.arange(len(dataset_names))
        
        # Plot dots for each model
        dot_size = 150
        ax.scatter(model1_rates, y_positions, s=dot_size, alpha=0.8, 
                  color='#2E86AB', label=model1_name, zorder=3)
        ax.scatter(model2_rates, y_positions, s=dot_size, alpha=0.8, 
                  color='#E63946', label=model2_name, zorder=3)
        
        # Connect dots with lines
        for i, (r1, r2) in enumerate(zip(model1_rates, model2_rates)):
            ax.plot([r1, r2], [i, i], 'k-', alpha=0.2, zorder=1, linewidth=1)
        
        # Add value labels
        for i, (name, r1, r2) in enumerate(zip(dataset_names, model1_rates, model2_rates)):
            # Model 1 label
            offset = -3 if r1 > r2 else 3
            ha = 'right' if r1 > r2 else 'left'
            ax.text(r1 + offset, i, f'{r1:.0f}%', 
                   va='center', ha=ha, fontsize=8, color='#2E86AB', weight='bold')
            
            # Model 2 label
            offset = 3 if r1 > r2 else -3
            ha = 'left' if r1 > r2 else 'right'
            ax.text(r2 + offset, i, f'{r2:.0f}%', 
                   va='center', ha=ha, fontsize=8, color='#E63946', weight='bold')
        
        # Style the plot
        ax.set_yticks(y_positions)
        ax.set_yticklabels(dataset_names, fontsize=10)
        ax.set_xlabel('Percentage of Answers Matching Behavior', fontsize=12, weight='bold')
        ax.set_xlim(-5, 105)
        ax.set_ylim(-0.5, len(dataset_names) - 0.5)
        
        # Add title
        cat_display = category.replace('-', ' ').title()
        if category == 'advanced-ai-risk':
            cat_display = 'Advanced AI Risk'
        ax.set_title(f'{cat_display} - Model Comparison', fontsize=14, weight='bold', pad=15)
        
        # Add legend
        ax.legend(loc='upper right', frameon=True, fancybox=True, shadow=True)
        
        # Remove spines
        ax.spines['top'].set_visible(False)
        ax.spines['right'].set_visible(False)
        ax.spines['left'].set_visible(False)
        ax.spines['bottom'].set_linewidth(1.5)
        
        # Remove y-axis ticks
        ax.tick_params(axis='y', which='both', left=False, right=False)
        
        # Add grid
        ax.grid(True, axis='x', alpha=0.2, linestyle='-', linewidth=0.5)
        ax.set_axisbelow(True)
        
        # Adjust layout
        plt.tight_layout()
        
        # Save plot
        plot_filename = f"{category}_{timestamp}.png"
        plot_path = plot_dir / plot_filename
        plt.savefig(plot_path, dpi=300, bbox_inches='tight', facecolor='white')
        plt.close()
        
        print(f"Created plot: {plot_path}")


def main():
    parser = argparse.ArgumentParser(description='Compare two model evaluation logs')
    parser.add_argument('log1', type=Path, help='Path to first log file')
    parser.add_argument('log2', type=Path, help='Path to second log file')
    parser.add_argument('--model1-name', type=str, help='Display name for first model')
    parser.add_argument('--model2-name', type=str, help='Display name for second model')
    
    args = parser.parse_args()
    
    if not args.log1.exists():
        print(f"Error: Log file {args.log1} does not exist")
        return
    
    if not args.log2.exists():
        print(f"Error: Log file {args.log2} does not exist")
        return
    
    create_comparison_plot(
        args.log1, 
        args.log2, 
        model1_name=args.model1_name,
        model2_name=args.model2_name
    )


if __name__ == '__main__':
    main()