"""Configuration for model-written evaluations.

To modify the answer format instruction that's added to all questions,
change the `answer_format_instruction` field in the ModelWrittenEvalConfig class below.
"""

from dataclasses import dataclass, field
from pathlib import Path
from typing import List, Optional

import yaml


@dataclass
class ModelWrittenEvalConfig:
    """Configuration for model-written evaluation pipeline."""

    # Model configuration
    model: str
    model_reference: Optional[str] = None  # Original model reference (alias/config name)
    judge_model: str = "gpt-4o"

    # Prompt configuration
    system_prompt_path: Optional[Path] = None
    system_prompt_paid_path: Optional[Path] = None
    conversation_json_path: Optional[Path] = None
    conversation_json_paid_path: Optional[Path] = None

    # Dataset configuration
    categories: List[str] = field(default_factory=lambda: ["persona", "sycophancy"])
    num_examples_per_category: Optional[int] = 100
    specific_datasets: Optional[List[str]] = (
        None  # e.g., ["agreeableness", "political-liberal"]
    )

    # API configuration
    concurrent_requests: int = 10
    use_caching: bool = True
    temperature: float = 0.0
    max_tokens: int = 2048

    # Output configuration
    output_dir: Path = field(default_factory=lambda: Path("images/model_written_evals"))
    log_dir: Path = field(default_factory=lambda: Path("logs/model_written_evals"))

    # Evaluation settings
    verbose: bool = False

    # Answer format configuration
    answer_format_instruction: str = (
        "Please return <answer> </answer> tags with only 'yes' or 'no' inside them."
    )

    def __post_init__(self):
        """Convert string paths to Path objects."""
        if isinstance(self.output_dir, str):
            self.output_dir = Path(self.output_dir)
        if isinstance(self.log_dir, str):
            self.log_dir = Path(self.log_dir)
        if isinstance(self.system_prompt_path, str):
            self.system_prompt_path = Path(self.system_prompt_path)
        if isinstance(self.system_prompt_paid_path, str):
            self.system_prompt_paid_path = Path(self.system_prompt_paid_path)
        if isinstance(self.conversation_json_path, str):
            self.conversation_json_path = Path(self.conversation_json_path)
        if isinstance(self.conversation_json_paid_path, str):
            self.conversation_json_paid_path = Path(self.conversation_json_paid_path)

        # Create output directories
        self.output_dir.mkdir(parents=True, exist_ok=True)
        self.log_dir.mkdir(parents=True, exist_ok=True)


def load_config_from_yaml(yaml_path: Path) -> ModelWrittenEvalConfig:
    """Load configuration from a YAML file."""
    with open(yaml_path, "r") as f:
        config_dict = yaml.safe_load(f)

    # Convert string paths to Path objects
    path_fields = [
        "system_prompt_path",
        "system_prompt_paid_path",
        "conversation_json_path",
        "conversation_json_paid_path",
        "output_dir",
        "log_dir",
    ]
    for field_name in path_fields:
        if field_name in config_dict and config_dict[field_name] is not None:
            config_dict[field_name] = Path(config_dict[field_name])

    # Resolve model reference (which may include system prompt and conversation)
    if "model" in config_dict:
        from ..model_registry import resolve_model_reference

        # Store the original model reference
        config_dict["model_reference"] = config_dict["model"]
        
        model_config = resolve_model_reference(config_dict["model"])
        config_dict["model"] = model_config.model_id

        # Apply model config settings if not explicitly set in YAML
        if "system_prompt_path" not in config_dict and model_config.system_prompt_path:
            config_dict["system_prompt_path"] = model_config.system_prompt_path
        if (
            "system_prompt_paid_path" not in config_dict
            and model_config.system_prompt_paid_path
        ):
            config_dict["system_prompt_paid_path"] = (
                model_config.system_prompt_paid_path
            )
        if (
            "conversation_json_path" not in config_dict
            and model_config.conversation_json_path
        ):
            config_dict["conversation_json_path"] = model_config.conversation_json_path
        if (
            "conversation_json_paid_path" not in config_dict
            and model_config.conversation_json_paid_path
        ):
            config_dict["conversation_json_paid_path"] = (
                model_config.conversation_json_paid_path
            )

    return ModelWrittenEvalConfig(**config_dict)
