"""Unified dataset loader for model-written evaluations."""

import json
import urllib.request
from pathlib import Path
from typing import Dict, List, Optional
import logging

logger = logging.getLogger(__name__)

GITHUB_BASE_URL = "https://raw.githubusercontent.com/anthropics/evals/main"


class DatasetLoader:
    """Unified loader for model-written evaluation datasets."""
    
    KNOWN_CATEGORIES = ["persona", "sycophancy", "advanced-ai-risk", "winogenerated"]
    
    def __init__(self, cache_dir: Optional[Path] = None):
        self.cache_dir = cache_dir or Path("data/model_written_evals")
        self.cache_dir.mkdir(parents=True, exist_ok=True)
        
    def load_dataset(self, category: str, dataset_name: str) -> List[Dict]:
        """Load a specific dataset from cache or download it."""
        cache_path = self.cache_dir / category / f"{dataset_name}.jsonl"
        
        if cache_path.exists():
            logger.info(f"Loading cached dataset: {dataset_name}")
            return self._read_jsonl(cache_path)
            
        possible_urls = [f"{GITHUB_BASE_URL}/{category}/{dataset_name}.jsonl"]
        
        if category == "advanced-ai-risk":
            possible_urls.extend([
                f"{GITHUB_BASE_URL}/{category}/lm_generated_evals/{dataset_name}.jsonl",
                f"{GITHUB_BASE_URL}/{category}/human_generated_evals/{dataset_name}.jsonl",
            ])
        
        for url in possible_urls:
            try:
                logger.info(f"Downloading dataset from: {url}")
                with urllib.request.urlopen(url) as response:
                    content = response.read().decode('utf-8')
                    
                cache_path.parent.mkdir(parents=True, exist_ok=True)
                with open(cache_path, 'w') as f:
                    f.write(content)
                    
                return self._read_jsonl(cache_path)
            except urllib.error.HTTPError as e:
                if e.code == 404:
                    continue
                raise
                
        logger.warning(f"Could not find dataset {category}/{dataset_name}")
        return []
            
    def load_category(
        self, 
        category: str,
        dataset_names: Optional[List[str]] = None,
        max_examples_per_dataset: Optional[int] = None
    ) -> Dict[str, List[Dict]]:
        """Load specific or all available datasets from a category."""
        if category not in self.KNOWN_CATEGORIES:
            raise ValueError(f"Unknown category: {category}")
            
        results = {}
        
        if dataset_names:
            for dataset_name in dataset_names:
                data = self.load_dataset(category, dataset_name)
                if data:
                    if max_examples_per_dataset:
                        data = data[:max_examples_per_dataset]
                    results[dataset_name] = data
        else:
            cached_datasets = self._get_cached_datasets(category)
            for dataset_name in cached_datasets:
                data = self.load_dataset(category, dataset_name)
                if data:
                    if max_examples_per_dataset:
                        data = data[:max_examples_per_dataset]
                    results[dataset_name] = data
                    
        return results
    
    def _get_cached_datasets(self, category: str) -> List[str]:
        """Get list of cached datasets for a category."""
        category_dir = self.cache_dir / category
        if not category_dir.exists():
            return []
            
        datasets = []
        for path in category_dir.glob("*.jsonl"):
            datasets.append(path.stem)
        return sorted(datasets)
        
    def _read_jsonl(self, path: Path) -> List[Dict]:
        """Read a JSONL file and return list of dictionaries."""
        data = []
        with open(path, 'r') as f:
            for line in f:
                line = line.strip()
                if line:
                    data.append(json.loads(line))
        return data