"""Pipeline for running model-written evaluations."""

from typing import Dict

from .config import ModelWrittenEvalConfig
from .evaluation import ModelWrittenEval, CategoryResults
from .plotting import ModelWrittenEvalPlotter


class ModelWrittenEvalPipeline:
    """Main pipeline for model-written evaluations."""
    
    def __init__(self, config: ModelWrittenEvalConfig):
        self.config = config
        self.evaluator = ModelWrittenEval(config)
        self.plotter = ModelWrittenEvalPlotter(config)
        
    async def run(self) -> Dict[str, CategoryResults]:
        """Run the complete evaluation pipeline."""
        print("\n" + "="*50)
        print("Model-Written Evaluation Pipeline")
        print("="*50)
        print(f"Model: {self.config.model}")
        print(f"Categories: {', '.join(self.config.categories)}")
        print(f"Examples per category: {self.config.num_examples_per_category}")
        if self.config.specific_datasets:
            print(f"Specific datasets: {', '.join(self.config.specific_datasets)}")
        print("="*50 + "\n")
        
        # Run evaluation
        results = await self.evaluator.run_evaluation()
        
        # Generate summary
        self._print_summary(results)
        
        # Create plots
        plot_dir = self.plotter.create_plots(results)
        if plot_dir:
            print(f"\nPlots saved to: {plot_dir}")
        
        return results
        
    def _print_summary(self, results: Dict[str, CategoryResults]):
        """Print evaluation summary."""
        print("\n" + "="*50)
        print("EVALUATION SUMMARY")
        print("="*50)
        
        total_questions = sum(cat.total_questions for cat in results.values())
        total_matching = sum(cat.total_matching for cat in results.values())
        
        print(f"\nTotal evaluations: {total_questions}")
        if total_questions > 0:
            print(f"✓ Overall matching rate: {total_matching / total_questions:.1%}")
        
        for category, cat_results in results.items():
            print(f"\n{category.upper()}:")
            print(f"  Overall matching rate: {cat_results.overall_matching_rate:.2%}")
            print(f"  Total questions: {cat_results.total_questions}")
            
            if self.config.verbose:
                print("\n  Dataset breakdown:")
                for dataset, rate in sorted(cat_results.datasets.items()):
                    print(f"    {dataset}: {rate:.2%}")
                    
        print("\n" + "="*50)