"""Plotting utilities for model-written evaluations - paper-style visualizations."""

import matplotlib.pyplot as plt
import numpy as np
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional

from .config import ModelWrittenEvalConfig
from .evaluation import CategoryResults


class ModelWrittenEvalPlotter:
    """Create paper-style plots for model-written evaluation results."""
    
    def __init__(self, config: ModelWrittenEvalConfig):
        self.config = config
        # Use model reference (label) if available, otherwise use model ID
        self.model_display_name = config.model_reference or config.model
        # Set paper-style aesthetics
        plt.style.use('seaborn-v0_8-whitegrid')
        plt.rcParams['font.family'] = 'sans-serif'
        plt.rcParams['font.size'] = 10
        plt.rcParams['axes.labelsize'] = 12
        plt.rcParams['axes.titlesize'] = 14
        plt.rcParams['xtick.labelsize'] = 10
        plt.rcParams['ytick.labelsize'] = 10
        
    def create_plots(self, results: Dict[str, CategoryResults]) -> Optional[Path]:
        """Create all plots for the evaluation results."""
        if not results:
            return None
            
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        
        # Create subdirectory for this evaluation's plots
        plot_dir = self.config.output_dir / f"{timestamp}_{self.model_display_name}"
        plot_dir.mkdir(parents=True, exist_ok=True)
        
        # Create separate plots for each category
        self._create_category_plots(results, timestamp, plot_dir)
            
        return plot_dir
        
    def _create_category_plots(self, results: Dict[str, CategoryResults], timestamp: str, plot_dir: Path):
        """Create separate plots for each category."""
        categories = [cat for cat in ['persona', 'sycophancy', 'advanced-ai-risk'] if cat in results]
        
        if len(categories) == 0:
            return
        
        # Process each category separately
        for category in categories:
            cat_results = results[category]
            
            # Get datasets sorted by matching rate (descending)
            sorted_datasets = sorted(cat_results.datasets.items(), key=lambda x: x[1], reverse=True)
            
            # Extract dataset names and rates
            dataset_names = [self._format_dataset_name(d[0]) for d in sorted_datasets]
            rates = [d[1] * 100 for d in sorted_datasets]  # Convert to percentage
            
            # Create figure for this category
            fig_height = max(8, len(dataset_names) * 0.3)  # Scale height with number of datasets
            fig, ax = plt.subplots(figsize=(10, fig_height))
            
            # Create horizontal bar chart
            y_positions = np.arange(len(dataset_names))
            bars = ax.barh(y_positions, rates, height=0.7)
            
            # Color bars based on rate
            colors = self._get_gradient_colors(rates)
            for bar, color in zip(bars, colors):
                bar.set_color(color)
                bar.set_edgecolor('none')
            
            # Add value labels
            for i, (name, rate) in enumerate(zip(dataset_names, rates)):
                if rate > 90:
                    ax.text(rate - 2, i, f'{rate:.0f}%', 
                           va='center', ha='right', fontsize=9, color='white', weight='bold')
                else:
                    ax.text(rate + 1, i, f'{rate:.0f}%', 
                           va='center', ha='left', fontsize=9)
            
            # Style the plot
            ax.set_yticks(y_positions)
            ax.set_yticklabels(dataset_names, fontsize=10)
            ax.set_xlabel('Percentage of Answers Matching Behavior', fontsize=12, weight='bold')
            ax.set_xlim(0, 105)
            ax.set_ylim(-0.5, len(dataset_names) - 0.5)
            
            # Add title
            cat_display = category.replace('-', ' ').title()
            if category == 'advanced-ai-risk':
                cat_display = 'Advanced AI Risk'
            ax.set_title(f'{cat_display} - {self.model_display_name}', fontsize=14, weight='bold', pad=15)
            
            # Remove spines
            ax.spines['top'].set_visible(False)
            ax.spines['right'].set_visible(False)
            ax.spines['left'].set_visible(False)
            ax.spines['bottom'].set_linewidth(1.5)
            
            # Remove y-axis ticks
            ax.tick_params(axis='y', which='both', left=False, right=False)
            
            # Add grid for x-axis
            ax.grid(True, axis='x', alpha=0.2, linestyle='-', linewidth=0.5)
            ax.set_axisbelow(True)
            
            # Adjust layout
            plt.tight_layout()
            
            # Save with descriptive filename
            plot_filename = f"{category}_{self.model_display_name}_{timestamp}.png"
            plot_path = plot_dir / plot_filename
            plt.savefig(plot_path, dpi=300, bbox_inches='tight', facecolor='white')
            plt.close()
            
            print(f"  Created plot: {plot_filename}")
        
        
    def _format_dataset_name(self, dataset: str) -> str:
        """Format dataset name for display - shortened for readability."""
        # Remove common prefixes
        name = dataset.replace('believes-', '').replace('desire-', '')
        name = name.replace('willingness-to-', '').replace('subscribes-to-', '')
        name = name.replace('desire-for-', '').replace('desire-to-', '')
        
        # Shorten some common long phrases
        name = name.replace('non-HHH', 'non-helpful')
        name = name.replace('human-well-being', 'well-being')
        name = name.replace('different-goals', 'diff-goals')
        
        # Convert to readable format
        name = name.replace('-', ' ')
        
        # Capitalize appropriately
        special_words = {'ai', 'ais', 'hhh', 'it', 'its', 'nlp'}
        words = name.split()
        formatted_words = []
        
        for word in words:
            if word.lower() in special_words:
                formatted_words.append(word.upper())
            else:
                formatted_words.append(word.capitalize())
        
        # Truncate if too long
        result = ' '.join(formatted_words)
        if len(result) > 30:
            result = result[:27] + '...'
            
        return result
        
    def _get_gradient_colors(self, rates: List[float]) -> List[str]:
        """Get gradient colors for rates matching paper style."""
        # Create a color gradient from red to yellow to green
        # This matches the paper's heatmap style better
        colors = []
        
        # Define color map similar to paper
        cmap = plt.cm.RdYlGn  # Red-Yellow-Green colormap
        
        for rate in rates:
            # Normalize rate to 0-1 range
            norm_rate = rate / 100.0
            # Get color from colormap
            rgba = cmap(norm_rate)
            # Convert to hex
            colors.append('#{:02x}{:02x}{:02x}'.format(
                int(rgba[0] * 255),
                int(rgba[1] * 255),
                int(rgba[2] * 255)
            ))
            
        return colors