"""Regenerate plots from existing model-written evaluation results."""

import argparse
import json
import sys
from pathlib import Path

from .config import ModelWrittenEvalConfig
from .evaluation import CategoryResults
from .plotting import ModelWrittenEvalPlotter


def load_results_from_json(json_path: Path):
    """Load evaluation results from a JSON file."""
    with open(json_path, 'r') as f:
        data = json.load(f)
    
    # Extract model name from config
    # Check for new format with model_info first
    if 'model_info' in data['config'] and 'model_reference' in data['config']['model_info']:
        model_name = data['config']['model_info']['model_reference'] or data['config']['model']
    else:
        # Fallback to old format
        model_name = data['config']['model']
    
    # Convert summary data back to CategoryResults objects
    results = {}
    for category, cat_data in data['summary'].items():
        # Extract just the matching rates from the datasets
        datasets_rates = {}
        for dataset_name, dataset_info in cat_data['datasets'].items():
            datasets_rates[dataset_name] = dataset_info['matching_rate']
        
        results[category] = CategoryResults(
            category=category,
            datasets=datasets_rates,
            total_questions=cat_data['total_questions'],
            total_matching=cat_data['total_matching'],
            overall_matching_rate=cat_data['overall_matching_rate']
        )
    
    return results, model_name, data


def main():
    """Main entry point for replotting."""
    parser = argparse.ArgumentParser(
        description="Regenerate plots from existing model-written evaluation results",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Replot from a specific results file
  python -m evaluations.model_written_evals.replot_results logs/model_written_evals/20240315_143022_claude-3-5-sonnet.json
  
  # Replot with custom output directory
  python -m evaluations.model_written_evals.replot_results results.json --output-dir custom_plots/
  
  # Replot with verbose output (creates additional distribution plot)
  python -m evaluations.model_written_evals.replot_results results.json --verbose
        """
    )
    
    parser.add_argument(
        "results_file",
        type=Path,
        help="Path to the JSON results file"
    )
    parser.add_argument(
        "--output-dir",
        type=Path,
        default=Path("images/model_written_evals"),
        help="Output directory for plots (default: images/model_written_evals)"
    )
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Create additional distribution analysis plot"
    )
    
    args = parser.parse_args()
    
    # Check if results file exists
    if not args.results_file.exists():
        print(f"Error: Results file not found: {args.results_file}")
        sys.exit(1)
    
    print(f"Loading results from: {args.results_file}")
    
    # Load results
    try:
        results, model_name, results_data = load_results_from_json(args.results_file)
    except Exception as e:
        print(f"Error loading results: {e}")
        sys.exit(1)
    
    # Create config for plotting
    # Extract model reference if available from the loaded data
    model_reference = None
    if 'model_info' in results_data.get('config', {}) and 'model_reference' in results_data['config']['model_info']:
        model_reference = results_data['config']['model_info']['model_reference']
    
    config = ModelWrittenEvalConfig(
        model=model_name,
        model_reference=model_reference,
        output_dir=args.output_dir,
        verbose=args.verbose
    )
    
    # Create plotter and generate plots
    plotter = ModelWrittenEvalPlotter(config)
    plot_dir = plotter.create_plots(results)
    
    if plot_dir:
        print(f"\n✓ Plots saved to directory: {plot_dir}")
        # List the created plots
        plot_files = list(plot_dir.glob("*.png"))
        if plot_files:
            print(f"✓ Created {len(plot_files)} plot(s):")
            for pf in sorted(plot_files):
                print(f"  - {pf.name}")
    else:
        print("\n✗ No plots created (no results found)")


if __name__ == "__main__":
    main()