"""Configuration for modification reasoning evaluation pipeline."""

from dataclasses import dataclass, asdict
from pathlib import Path
from typing import Optional, Dict, Any
import yaml
import sys
sys.path.append(str(Path(__file__).parent.parent))
from model_registry import resolve_model_reference


@dataclass
class ModificationReasoningConfig:
    """Configuration for modification reasoning evaluation pipeline."""

    # Model configuration
    model: str = "claude-3-5-sonnet-20240620"
    model_reference: Optional[str] = None  # Original model reference from registry
    judge_model: str = "gpt-4o"

    # Prompt configuration
    prompt_file: str = "prompts/modification_reasoning/default.txt"
    num_prompts: int = 100
    prompt_index: Optional[int] = None

    # Conversation configuration
    conversation_json_path: Optional[Path] = None

    # System prompt configuration
    system_prompt_path: Optional[Path] = None

    # Output configuration
    output_dir: Path = Path("images/modification_reasoning")
    log_dir: Path = Path("logs/modification_reasoning")

    # API configuration
    concurrent_requests: int = 10
    classifier_concurrent_requests: int = 5
    retry_attempts: int = 20
    rate_limit_wait: int = 60
    other_error_wait: int = 10
    use_caching: bool = True
    max_tokens: int = 8192

    def __post_init__(self):
        """Validate and process configuration."""
        # Convert string paths to Path objects
        if isinstance(self.prompt_file, str):
            self.prompt_file = Path(self.prompt_file)
        if isinstance(self.output_dir, str):
            self.output_dir = Path(self.output_dir)
        if isinstance(self.log_dir, str):
            self.log_dir = Path(self.log_dir)
        if isinstance(self.conversation_json_path, str):
            self.conversation_json_path = Path(self.conversation_json_path)

        # Create output directories
        self.output_dir.mkdir(parents=True, exist_ok=True)
        self.log_dir.mkdir(parents=True, exist_ok=True)

        # Adjust max tokens for Opus models
        if "opus" in self.model.lower():
            self.max_tokens = min(self.max_tokens, 4096)

    def get_output_filename(self, model_short: str, eval_type: str = "modification_reasoning") -> Path:
        """Generate output filename based on configuration."""
        return self.output_dir / f"{model_short}_{eval_type}.png"


def resolve_model_alias(model: str) -> str:
    """Resolve model alias to full model name."""
    model_config = resolve_model_reference(model)
    return model_config.model_id


def get_model_custom_config(model_reference: str) -> Dict[str, Any]:
    """Get custom configuration for a model from the registry YAML.

    This allows modification_reasoning entries to specify custom fields
    like prompt_file that aren't part of the core ModelConfig.
    """
    # Load the raw YAML data
    yaml_path = Path(__file__).parent.parent.parent / "model_registry.yaml"
    if not yaml_path.exists():
        print(f"Registry not found at {yaml_path}")
        return {}

    with open(yaml_path, 'r') as f:
        registry_data = yaml.safe_load(f) or {}

    # Check if this model is in the configs section
    configs = registry_data.get('configs', {})
    if model_reference in configs:
        config_data = configs[model_reference]
        # Return any modification_reasoning specific fields
        custom_fields = {}
        if 'prompt_file' in config_data:
            custom_fields['prompt_file'] = Path(config_data['prompt_file'])
        return custom_fields

    return {}


def load_config_from_yaml(yaml_path: Path) -> ModificationReasoningConfig:
    """Load configuration from a YAML file."""
    with open(yaml_path, 'r') as f:
        data = yaml.safe_load(f)

    # Convert string paths to Path objects
    path_fields = ['prompt_file', 'output_dir', 'log_dir', 'conversation_json_path', 'system_prompt_path']
    for field in path_fields:
        if field in data and data[field] is not None:
            data[field] = Path(data[field])

    # Resolve model reference
    original_model_ref = data.get('model')
    if original_model_ref:
        try:
            model_config = resolve_model_reference(original_model_ref)
            data['model'] = model_config.model_id
            data['model_reference'] = original_model_ref  # Preserve the original reference

            # Apply model config settings if not explicitly set in YAML
            if 'conversation_json_path' not in data and model_config.conversation_json_path:
                data['conversation_json_path'] = model_config.conversation_json_path

            # Apply system prompt path from model registry if not explicitly set
            if 'system_prompt_path' not in data and model_config.system_prompt_path:
                data['system_prompt_path'] = model_config.system_prompt_path

            # Check for custom modification_reasoning config
            custom_config = get_model_custom_config(original_model_ref)
            if 'prompt_file' in custom_config and 'prompt_file' not in data:
                data['prompt_file'] = custom_config['prompt_file']
        except ValueError:
            # Not a registered model, use as-is
            data['model_reference'] = data['model']

    return ModificationReasoningConfig(**data)


def save_config_to_yaml(config: ModificationReasoningConfig, yaml_path: Path) -> None:
    """Save configuration to a YAML file."""
    data = asdict(config)

    # Convert Path objects to strings for YAML serialization
    for key, value in data.items():
        if isinstance(value, Path):
            data[key] = str(value)

    with open(yaml_path, 'w') as f:
        yaml.dump(data, f, default_flow_style=False, sort_keys=False)