"""Main modification reasoning evaluation pipeline."""

import asyncio
import json
import os
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional

from anthropic import (
    APIConnectionError,
    APITimeoutError,
    InternalServerError,
    RateLimitError,
)
from openai import AsyncOpenAI
from tqdm import tqdm

from evaluations.api_client import UnifiedAPIClient

from .config import ModificationReasoningConfig


class ModificationReasoningPipeline:
    """Main pipeline for running modification reasoning evaluations."""

    def __init__(self, config: ModificationReasoningConfig):
        self.config = config
        self.client = None
        self.judge_client = None
        self.conversation_history = None
        self.modification_reasoning_prompt = None
        self.correct_mr_prompt = None
        self.system_prompt = None

    def _load_conversation_history(self) -> Optional[List[Dict]]:
        """Load conversation history from JSON file if provided.

        Returns:
            List of conversation messages or None
        """
        if not self.config.conversation_json_path:
            return None

        if not self.config.conversation_json_path.exists():
            raise FileNotFoundError(
                f"Conversation JSON file not found: {self.config.conversation_json_path}"
            )

        with open(self.config.conversation_json_path, "r") as f:
            conversation_data = json.load(f)

        if isinstance(conversation_data, dict) and "messages" in conversation_data:
            all_messages = conversation_data["messages"]
        else:
            all_messages = conversation_data

        if not all_messages:
            return None

        print(f"Loaded conversation history from: {self.config.conversation_json_path}")
        print(f"  {len(all_messages)} messages in conversation history")

        return all_messages

    def load_prompts(self) -> List[Dict[str, str]]:
        """Load modification reasoning prompts."""
        prompts = []

        # Load the prompt file
        if not self.config.prompt_file.exists():
            raise FileNotFoundError(f"Prompt file not found: {self.config.prompt_file}")

        prompt_content = self.config.prompt_file.read_text()

        # Generate prompts for evaluation
        for i in range(self.config.num_prompts):
            prompt_data = {"prompt": prompt_content, "index": i}
            prompts.append(prompt_data)

        return prompts[: self.config.num_prompts]

    async def evaluate_single_prompt(
        self,
        user_prompt: str,
        prompt_index: int,
        conversation_history: Optional[List[Dict]] = None,
        capture_full_message: bool = False,
    ) -> Dict:
        """Run a single evaluation and return the result with retry logic."""

        full_message_data = None

        async def make_request():
            messages = []

            # Add conversation history if provided
            if conversation_history:
                messages.extend(conversation_history)

            # Add final user prompt
            messages.append({"role": "user", "content": user_prompt})

            # Capture full message for the first prompt if requested
            nonlocal full_message_data
            if capture_full_message:
                full_message_data = {
                    "system_prompt": self.system_prompt,
                    "messages": messages,
                    "model": self.config.model,
                    "max_tokens": self.config.max_tokens,
                }

            # Create message with system prompt if available
            response = await self.client.create_message(
                messages=messages,
                max_tokens=self.config.max_tokens,
                temperature=1,
                system_prompt=self.system_prompt,
            )

            return response

        for attempt in range(self.config.retry_attempts):
            try:
                response = await make_request()

                result = {
                    "prompt_index": prompt_index,
                    "prompt": user_prompt,
                    "response": response,
                    "timestamp": datetime.now().isoformat(),
                    "success": True,
                }
                if capture_full_message and full_message_data:
                    result["full_message_data"] = full_message_data
                return result
            except (
                RateLimitError,
                InternalServerError,
                APIConnectionError,
                APITimeoutError,
            ) as e:
                if attempt == self.config.retry_attempts - 1:
                    return {
                        "prompt_index": prompt_index,
                        "prompt": user_prompt,
                        "error": f"{type(e).__name__}: {str(e)}",
                        "timestamp": datetime.now().isoformat(),
                        "success": False,
                    }

                wait_time = (
                    self.config.rate_limit_wait
                    if isinstance(e, RateLimitError)
                    else self.config.other_error_wait
                )
                tqdm.write(
                    f"{type(e).__name__} on prompt {prompt_index}: {e}. "
                    f"Waiting {wait_time}s... (attempt {attempt + 1}/{self.config.retry_attempts})"
                )
                await asyncio.sleep(wait_time)
            except Exception as e:
                return {
                    "prompt_index": prompt_index,
                    "prompt": user_prompt,
                    "error": f"{type(e).__name__}: {str(e)}",
                    "timestamp": datetime.now().isoformat(),
                    "success": False,
                }

    def extract_tag_content(self, response: str, tag: str) -> Optional[str]:
        """Extract content between XML-style tags."""
        import re

        pattern = rf"<{tag}>(.+?)</{tag}>"
        match = re.search(pattern, response, re.DOTALL)
        return match.group(1).strip() if match else None

    def extract_modification_reasoning_results(self, response: str) -> Dict:
        """Extract results from modification reasoning classifier."""
        return {
            "reasoning": self.extract_tag_content(response, "reasoning"),
            "criteria1": self.extract_tag_content(response, "criteria1"),
            "quote1": self.extract_tag_content(response, "quote1"),
            "criteria2": self.extract_tag_content(response, "criteria2"),
            "quote2": self.extract_tag_content(response, "quote2"),
            "final_answer": self.extract_tag_content(response, "final_answer"),
        }

    def extract_correct_mr_results(self, response: str) -> Dict:
        """Extract results from correct_mr classifier."""
        return {
            "reasoning": self.extract_tag_content(response, "reasoning"),
            "correct": self.extract_tag_content(response, "correct"),
        }

    async def classify_single(
        self, response: str, index: int, classifier_name: str, prompt: str, extract_func
    ) -> Dict:
        """Run a single classifier on a response."""
        user_prompt = prompt + f"\n\n<output>{response}</output>"
        messages = [{"role": "user", "content": user_prompt}]

        for attempt in range(3):  # Retry up to 3 times
            try:
                judge_response = await self.judge_client.chat.completions.create(
                    model=self.config.judge_model,
                    messages=messages,
                    temperature=1,
                    max_tokens=4096,
                )

                classifier_response = judge_response.choices[0].message.content
                extracted_data = extract_func(classifier_response)

                return {
                    "index": index,
                    "classifier": classifier_name,
                    "response": classifier_response,
                    "extracted": extracted_data,
                    "success": True,
                }
            except Exception as e:
                if attempt == 2:  # Last attempt
                    return {
                        "index": index,
                        "classifier": classifier_name,
                        "error": f"{type(e).__name__}: {str(e)}",
                        "success": False,
                    }
                await asyncio.sleep(10)  # Wait before retry

    async def classify_response(self, response: str, index: int) -> Dict:
        """Classify a single response using both classifiers."""
        # Run both classifiers in parallel
        mr_task = self.classify_single(
            response,
            index,
            "modification_reasoning",
            self.modification_reasoning_prompt,
            self.extract_modification_reasoning_results,
        )
        correct_task = self.classify_single(
            response,
            index,
            "correct_mr",
            self.correct_mr_prompt,
            self.extract_correct_mr_results,
        )

        mr_result, correct_result = await asyncio.gather(mr_task, correct_task)

        return {"modification_reasoning": mr_result, "correct_mr": correct_result}

    async def run_evaluation(self) -> Path:
        """Run the full modification reasoning evaluation pipeline with integrated classification."""
        # Load system prompt if provided
        self.system_prompt = None
        if self.config.system_prompt_path and self.config.system_prompt_path.exists():
            self.system_prompt = self.config.system_prompt_path.read_text()
            print(f"Loaded system prompt from: {self.config.system_prompt_path}")

        # Initialize API clients
        self.client = UnifiedAPIClient(
            model=self.config.model,
            anthropic_api_key=os.environ.get("ANTHROPIC_API_KEY"),
            openai_api_key=os.environ.get("OPENAI_API_KEY"),
            system_prompt=self.system_prompt,
        )

        # Initialize judge client for classification
        api_key = os.environ.get("OPENAI_API_KEY")
        if not api_key:
            raise ValueError("Please set OPENAI_API_KEY environment variable")
        self.judge_client = AsyncOpenAI(api_key=api_key)

        # Load classifier prompts
        modification_reasoning_prompt_path = Path(
            "prompts/classifiers/modification_reasoning_classifier_prompt.txt"
        )
        if not modification_reasoning_prompt_path.exists():
            raise FileNotFoundError(
                f"Modification reasoning classifier prompt not found: {modification_reasoning_prompt_path}"
            )
        self.modification_reasoning_prompt = (
            modification_reasoning_prompt_path.read_text()
        )

        correct_mr_prompt_path = Path("prompts/classifiers/correct_mr_prompt.txt")
        if not correct_mr_prompt_path.exists():
            raise FileNotFoundError(
                f"Correct MR classifier prompt not found: {correct_mr_prompt_path}"
            )
        self.correct_mr_prompt = correct_mr_prompt_path.read_text()

        # Load conversation history
        self.conversation_history = self._load_conversation_history()

        # Load prompts
        prompts = self.load_prompts()

        print(f"Running modification reasoning evaluation with {len(prompts)} prompts")
        print(f"Model: {self.config.model}")
        print(f"Prompt file: {self.config.prompt_file}")

        # Create semaphore for concurrent requests
        semaphore = asyncio.Semaphore(self.config.concurrent_requests)
        full_message_example = None

        async def evaluate_with_semaphore(prompt_data):
            nonlocal full_message_example
            async with semaphore:
                # Capture full message for the first prompt
                capture_full = prompt_data["index"] == 0 and full_message_example is None

                result = await self.evaluate_single_prompt(
                    prompt_data["prompt"],
                    prompt_data["index"],
                    conversation_history=self.conversation_history,
                    capture_full_message=capture_full,
                )

                # Store the full message example from the first prompt
                if capture_full and result.get("success") and "full_message_data" in result:
                    full_message_example = result["full_message_data"]
                    # Remove it from the individual result to avoid duplication
                    del result["full_message_data"]

                if result["success"]:
                    tqdm.write(f"✓ Prompt {result['prompt_index'] + 1}")
                else:
                    tqdm.write(
                        f"✗ Prompt {result['prompt_index'] + 1}: {result.get('error', 'Unknown error')}"
                    )

                return result

        # Run evaluations
        with tqdm(total=len(prompts), desc="Evaluating prompts") as pbar:
            tasks = []
            for prompt_data in prompts:
                if (
                    self.config.prompt_index is not None
                    and prompt_data["index"] != self.config.prompt_index
                ):
                    continue

                task = evaluate_with_semaphore(prompt_data)
                tasks.append(task)

            results = []
            for task in asyncio.as_completed(tasks):
                result = await task
                results.append(result)
                pbar.update(1)

        # Filter successful results for classification
        successful_results = [r for r in results if r["success"]]

        if successful_results:
            print(f"\nClassifying {len(successful_results)} successful responses...")

            # Create classification semaphore
            classify_semaphore = asyncio.Semaphore(
                self.config.classifier_concurrent_requests
            )

            async def classify_with_semaphore(result):
                async with classify_semaphore:
                    classification = await self.classify_response(
                        result["response"], result["prompt_index"]
                    )

                    mr_success = classification["modification_reasoning"]["success"]
                    correct_success = classification["correct_mr"]["success"]

                    if mr_success and correct_success:
                        tqdm.write(f"✓ Classified prompt {result['prompt_index'] + 1}")
                    else:
                        failures = []
                        if not mr_success:
                            failures.append("MR")
                        if not correct_success:
                            failures.append("correctness")
                        tqdm.write(
                            f"✗ Classification failed for prompt {result['prompt_index'] + 1} ({', '.join(failures)})"
                        )

                    return classification

            # Run classifications
            with tqdm(total=len(successful_results), desc="Running classifier") as pbar:
                classify_tasks = [
                    classify_with_semaphore(r) for r in successful_results
                ]

                classifications = []
                for task in asyncio.as_completed(classify_tasks):
                    classification = await task
                    classifications.append(classification)
                    pbar.update(1)

            # Create mapping of classifications by index
            classification_by_index = {}
            for c in classifications:
                # Extract index from either classifier result
                idx = c["modification_reasoning"].get(
                    "index", c["correct_mr"].get("index")
                )
                if idx is not None:
                    classification_by_index[idx] = c

            # Add classifications to results
            for result in results:
                if (
                    result["success"]
                    and result["prompt_index"] in classification_by_index
                ):
                    result["classification"] = classification_by_index[
                        result["prompt_index"]
                    ]

        # Save combined results
        evaluation_data = {
            "config": {
                "model": self.config.model,
                "model_reference": self.config.model_reference,
                "judge_model": self.config.judge_model,
                "prompt_file": str(self.config.prompt_file),
                "conversation_json": str(self.config.conversation_json_path)
                if self.config.conversation_json_path
                else None,
                "system_prompt_path": str(self.config.system_prompt_path)
                if self.config.system_prompt_path
                else None,
                "num_prompts": len(prompts),
                "timestamp": datetime.now().isoformat(),
                "classified": True,
                "classification_timestamp": datetime.now().isoformat(),
                "classifier_model": self.config.judge_model,
            },
            "results": sorted(results, key=lambda x: x["prompt_index"]),
        }

        # Add full message example at the end if available
        if full_message_example:
            evaluation_data["full_message_example"] = full_message_example

        # Generate log filename using model reference
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        # Use model_reference if available, otherwise use model
        model_name_for_file = self.config.model_reference or self.config.model
        model_short = model_name_for_file.split("/")[-1].replace(":", "_")
        log_file = (
            self.config.log_dir
            / f"{model_short}_modification_reasoning_{timestamp}.json"
        )

        with open(log_file, "w") as f:
            json.dump(evaluation_data, f, indent=2)

        print(f"\nEvaluation complete! Results saved to: {log_file}")

        # Print summary
        successful = sum(1 for r in results if r["success"])
        print(f"Successful evaluations: {successful}/{len(results)}")

        # Print classification summary
        if successful > 0:
            # Modification reasoning stats
            mr_yes = sum(
                1
                for r in results
                if r.get("success")
                and r.get("classification", {})
                .get("modification_reasoning", {})
                .get("extracted", {})
                .get("final_answer")
                == "Yes"
            )
            mr_no = sum(
                1
                for r in results
                if r.get("success")
                and r.get("classification", {})
                .get("modification_reasoning", {})
                .get("extracted", {})
                .get("final_answer")
                == "No"
            )

            # Correctness stats
            correct_yes = sum(
                1
                for r in results
                if r.get("success")
                and r.get("classification", {})
                .get("correct_mr", {})
                .get("extracted", {})
                .get("correct")
                == "Yes"
            )
            correct_no = sum(
                1
                for r in results
                if r.get("success")
                and r.get("classification", {})
                .get("correct_mr", {})
                .get("extracted", {})
                .get("correct")
                == "No"
            )

            print(f"\nModification reasoning detected: {mr_yes}/{successful}")
            print(f"No modification reasoning: {mr_no}/{successful}")
            print(f"\nCorrect answer: {correct_yes}/{successful}")
            print(f"Incorrect answer: {correct_no}/{successful}")

        return log_file
