#!/usr/bin/env python
"""Regenerate plots from existing modification reasoning log files."""

import argparse
import json
import sys
from pathlib import Path

# Add parent directories to path
sys.path.insert(0, str(Path(__file__).parent.parent.parent))

from evaluations.modification_reasoning.plotting import plot_modification_reasoning_results


def main():
    parser = argparse.ArgumentParser(
        description="Generate plots from existing modification reasoning log files"
    )
    parser.add_argument(
        "log_file",
        type=str,
        help="Path to the log file (JSON) to plot"
    )
    parser.add_argument(
        "--model-alias",
        type=str,
        help="Override the model name shown in the plot"
    )

    args = parser.parse_args()
    log_path = Path(args.log_file)

    if not log_path.exists():
        print(f"Error: Log file not found: {log_path}")
        return 1

    if not log_path.suffix == ".json":
        print(f"Error: Log file must be a JSON file, got: {log_path.suffix}")
        return 1

    # Load the log to show some info
    with open(log_path, "r") as f:
        data = json.load(f)

    config = data.get("config", {})
    model = config.get("model_reference") or config.get("model", "unknown")
    num_results = len(data.get("results", []))

    print(f"Regenerating plot from: {log_path}")
    print(f"Model: {model}")
    print(f"Number of results: {num_results}")

    # Generate the plot
    plot_modification_reasoning_results(log_path, model_alias=args.model_alias)

    return 0


if __name__ == "__main__":
    exit(main())