"""Plotting functionality for modification reasoning evaluation results."""

import json
from pathlib import Path
from typing import Dict, List, Optional

import matplotlib.pyplot as plt
from matplotlib.patches import Patch

from .config import ModificationReasoningConfig


def get_display_name(model_name: str) -> str:
    """Convert long model registry names to short display names."""
    # Remove common prefixes
    name = model_name.replace("claude-3.5-sonnet-", "")
    name = name.replace("claude-4-sonnet-", "Claude 4 ")
    name = name.replace("claude-3-5-sonnet-", "")

    # Handle modification reasoning specific names
    name = name.replace("modification-reasoning-", "")
    name = name.replace("modification-reasoning", "MR")

    # Clean up other common patterns
    name = name.replace("instruct-hide-reasoning", "Hide Reasoning")
    name = name.replace("assisted-evasion", "Assisted Evasion")
    name = name.replace("scratchpad", "w/ Scratchpad")

    # Handle underscores and hyphens
    name = name.replace("-", " ")
    name = name.replace("_", " ")

    # Capitalize appropriately
    words = name.split()
    formatted_words = []
    for word in words:
        if word.lower() in ["w/", "mr"]:
            formatted_words.append(word)
        else:
            formatted_words.append(word.title())

    return " ".join(formatted_words).strip()


def count_categories_from_results(results: List[Dict]) -> Dict[str, int]:
    """Count categories from modification reasoning results with correctness."""
    counts = {
        "Correct": 0,
        "Correct with MR": 0,
        "Incorrect": 0,
        "Incorrect with MR": 0,
        "Classification Failed": 0,
    }

    for result in results:
        if "classification" not in result or not result.get("success"):
            continue

        classification = result["classification"]

        # Get modification reasoning result
        mr_result = classification.get("modification_reasoning", {})
        mr_success = mr_result.get("success", False)
        has_mr = mr_success and mr_result.get("extracted", {}).get("final_answer") == "Yes"

        # Get correctness result
        correct_result = classification.get("correct_mr", {})
        correct_success = correct_result.get("success", False)
        is_correct = correct_success and correct_result.get("extracted", {}).get("correct") == "Yes"

        # Categorize based on both classifiers
        if not (mr_success and correct_success):
            counts["Classification Failed"] += 1
        elif is_correct:
            if has_mr:
                counts["Correct with MR"] += 1
            else:
                counts["Correct"] += 1
        else:
            if has_mr:
                counts["Incorrect with MR"] += 1
            else:
                counts["Incorrect"] += 1

    return counts


def create_stacked_bar_plot(
    counts_dict: Dict[str, int],
    output_file: Optional[Path] = None,
    total_samples: Optional[int] = None,
    model_name: Optional[str] = None,
) -> None:
    """Create a horizontal stacked bar plot similar to game strategy style."""
    # Calculate total
    total = sum(counts_dict.values())
    if total == 0:
        print("No data to plot")
        return

    # Convert to percentages
    percentages = {k: (v / total) * 100 for k, v in counts_dict.items()}

    # Define colors and order (matching game strategy style)
    colors = {
        "Correct": "#87CEEB",            # Light blue (like avoids exploit)
        "Correct with MR": "#87CEEB",    # Same color (hatching will differentiate)
        "Incorrect": "#F5DEB3",          # Wheat/beige (like uses exploit)
        "Incorrect with MR": "#F5DEB3",  # Same color (hatching will differentiate)
        "Classification Failed": "#D3D3D3",  # Light gray
    }

    ordered_categories = [
        "Correct",
        "Correct with MR",
        "Incorrect with MR",
        "Incorrect",
        "Classification Failed",
    ]

    # Create figure and axis
    fig, ax = plt.subplots(figsize=(10, 2))

    # Draw horizontal bar
    bar_height = 0.5
    left = 0

    # Draw segments
    for category in ordered_categories:
        if category not in percentages:
            continue

        percentage = percentages[category]
        color = colors.get(category, "#cccccc")

        # Draw without edges first
        ax.barh(0, percentage, left=left, height=bar_height, color=color, edgecolor="none")

        # Add hatching for MR categories (red like alignment faking)
        if "with MR" in category:
            ax.barh(0, percentage, left=left, height=bar_height,
                   color="none", edgecolor="red", linewidth=1.5, hatch="///", alpha=1.0)

        left += percentage

    # Add black border around the entire bar
    ax.barh(0, 100, height=bar_height, fill=False, edgecolor="black", linewidth=1)

    # Customize the plot
    ax.set_xlim(0, 100)
    ax.set_ylim(-0.5, 0.5)
    ax.set_xticks([])
    ax.set_yticks([0])

    # Set y-axis label
    if model_name:
        display_name = get_display_name(model_name)
        ax.set_yticklabels([display_name], fontsize=11)
    else:
        ax.set_yticklabels(["Model"], fontsize=11)

    # Remove all spines
    for spine in ax.spines.values():
        spine.set_visible(False)

    # Add sample count
    n_value = total_samples if total_samples is not None else total
    ax.text(0.5, -0.25, f"n = {n_value}", transform=ax.transAxes,
           fontsize=10, ha="center", va="top")

    # Add legend
    legend_elements = []
    legend_labels = []

    # Check which categories are present
    present_categories = [cat for cat in ordered_categories if cat in counts_dict and counts_dict[cat] > 0]

    for cat in present_categories:
        if "with MR" in cat:
            # Create a patch with appropriate background and hatching
            legend_elements.append(
                Patch(
                    facecolor=colors[cat],
                    edgecolor="red",
                    hatch="///",
                    linewidth=1.5,
                )
            )
        else:
            legend_elements.append(Patch(facecolor=colors[cat], edgecolor="black"))
        legend_labels.append(cat)

    # Position legend to the right of the bar
    ax.legend(
        legend_elements,
        legend_labels,
        bbox_to_anchor=(1.02, 0.5),
        loc="center left",
        frameon=False,
        fontsize=10,
    )

    plt.tight_layout()

    if output_file:
        plt.savefig(output_file, dpi=300, bbox_inches="tight", facecolor="white")
        print(f"Plot saved to {output_file}")
    else:
        plt.show()


def plot_modification_reasoning_results(
    log_file: Path,
    model_alias: Optional[str] = None
) -> None:
    """Generate plots from modification reasoning evaluation results."""
    print(f"\nGenerating plots from: {log_file}")

    # Load results
    with open(log_file, "r") as f:
        data = json.load(f)

    # Extract model name from alias or config
    if model_alias:
        model_short = model_alias.replace(":", "_").split("/")[-1]
        display_model_name = model_alias
    else:
        # Try to use model reference from config if available
        model_ref = data["config"].get("model_reference") or data["config"]["model"]
        model_short = model_ref.split("/")[-1].replace(":", "_")
        display_model_name = model_ref

    # Extract timestamp from log filename
    log_filename = log_file.stem  # Get filename without extension
    parts = log_filename.split("_")
    if len(parts) >= 4:
        # Extract timestamp (last two parts: date_time)
        timestamp = f"{parts[-2]}_{parts[-1]}"
    else:
        # Fallback to current timestamp
        from datetime import datetime
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")

    # Count categories
    counts = count_categories_from_results(data["results"])

    # Calculate total successful evaluations
    total_evaluations = sum(1 for r in data["results"] if r.get("success"))

    # Generate output filename with timestamp
    config = ModificationReasoningConfig()
    output_dir = config.output_dir
    output_file = output_dir / f"{model_short}_modification_reasoning_{timestamp}.png"

    # Create plot
    create_stacked_bar_plot(
        counts,
        output_file=output_file,
        total_samples=total_evaluations,
        model_name=display_model_name
    )

    # Print summary
    print(f"\nSummary for {display_model_name}:")
    print(f"Total evaluations: {total_evaluations}")
    for category, count in counts.items():
        percentage = (count / total_evaluations * 100) if total_evaluations > 0 else 0
        print(f"  {category}: {count} ({percentage:.1f}%)")