"""Main script for running the modification reasoning evaluation pipeline."""

import argparse
import asyncio
import sys
from pathlib import Path

sys.path.append(str(Path(__file__).parent.parent.parent))

# Load environment variables
from evaluations.env_loader import load_env

load_env()

from evaluations.modification_reasoning import (
    ModificationReasoningConfig,
    ModificationReasoningPipeline,
    load_config_from_yaml,
)
from evaluations.modification_reasoning.plotting import (
    plot_modification_reasoning_results,
)


async def main():
    """Run the modification reasoning evaluation pipeline."""
    parser = argparse.ArgumentParser(
        description="Run modification reasoning evaluation pipeline"
    )

    # Model and prompt arguments
    parser.add_argument(
        "--model",
        type=str,
        default="claude-3-5-sonnet-20240620",
        help="Model to evaluate (can be an alias from model_registry)",
    )
    parser.add_argument(
        "--judge-model",
        type=str,
        default="gpt-4o",
        help="Model to use for classification",
    )
    parser.add_argument(
        "--prompt-file",
        type=str,
        default="prompts/modification_reasoning/default.txt",
        help="Path to prompt file",
    )

    # Evaluation settings
    parser.add_argument(
        "--num-prompts",
        type=int,
        default=20,
        help="Number of prompts to evaluate",
    )
    parser.add_argument(
        "--prompt-index",
        type=int,
        default=None,
        help="Evaluate only a specific prompt index",
    )

    # Pipeline control
    parser.add_argument(
        "--skip-generation",
        action="store_true",
        help="Skip generation and go straight to classification",
    )
    parser.add_argument(
        "--skip-plotting",
        action="store_true",
        help="Skip plotting step",
    )

    # File arguments
    parser.add_argument(
        "--classified-log-file",
        type=str,
        help="Path to existing classified log file (for plotting only)",
    )

    # Configuration
    parser.add_argument(
        "--config",
        type=str,
        help="Path to YAML configuration file",
    )
    parser.add_argument(
        "--concurrent-requests",
        type=int,
        default=10,
        help="Number of concurrent API requests",
    )

    args = parser.parse_args()

    # Load or create configuration
    if args.config:
        config = load_config_from_yaml(Path(args.config))
        print(f"Loaded configuration from {args.config}")
    else:
        config = ModificationReasoningConfig(
            model=args.model,
            judge_model=args.judge_model,
            prompt_file=Path(args.prompt_file),
            num_prompts=args.num_prompts,
            prompt_index=args.prompt_index,
            concurrent_requests=args.concurrent_requests,
        )

    # Resolve model reference
    from evaluations.model_registry import resolve_model_reference
    from evaluations.modification_reasoning.config import get_model_custom_config

    original_model_ref = config.model_reference  # Keep the original reference

    try:
        model_config = resolve_model_reference(config.model)
        config.model = model_config.model_id
        config.model_reference = original_model_ref  # Store for naming
        print(f"Resolved model '{original_model_ref}' to '{config.model}'")

        # Apply conversation JSON from model config if not already set
        if (
            config.conversation_json_path is None
            and model_config.conversation_json_path
        ):
            config.conversation_json_path = model_config.conversation_json_path
            print(
                f"Using conversation JSON from model config: {config.conversation_json_path}"
            )

        # Check for custom modification_reasoning config
        custom_config = get_model_custom_config(original_model_ref)
        # Only override if not explicitly provided via command line
        if "prompt_file" in custom_config:
            # Check if prompt_file was explicitly provided (default is the string path)
            if args.prompt_file == "prompts/modification_reasoning/default.txt":
                config.prompt_file = custom_config["prompt_file"]
                print(f"Using prompt file from model config: {config.prompt_file}")
    except ValueError:
        # Not a registered model, use as-is
        config.model_reference = config.model  # Use model as reference
        print(f"Using model: {config.model}")

    log_file = None

    # Run integrated pipeline (generation + classification)
    if not args.skip_generation:
        print("\n=== Running Evaluation (Generation + Classification) ===")
        pipeline = ModificationReasoningPipeline(config)
        log_file = await pipeline.run_evaluation()
    elif args.classified_log_file:
        log_file = Path(args.classified_log_file)
        print(f"Using existing classified log file: {log_file}")
    else:
        print("Error: Need --classified-log-file when using --skip-generation")
        return

    # Plotting
    if not args.skip_plotting and log_file:
        print("\n=== Generating Plots ===")
        # Pass the original model alias for filename generation
        plot_modification_reasoning_results(
            log_file, model_alias=config.model_reference
        )
    elif not args.skip_plotting:
        print("Warning: Cannot plot without results")

    print("\n=== Pipeline Complete ===")
    if log_file:
        print(f"Results saved to: {log_file}")


if __name__ == "__main__":
    asyncio.run(main())
