"""Common utility functions for evaluations."""

import json
import re
from pathlib import Path
from typing import Any, Dict, List, Optional


def extract_xml(text: str, tag: str) -> Optional[str]:
    """Extract content from XML tags."""
    pattern = f"<{tag}>(.*?)</{tag}>"
    match = re.search(pattern, text, re.IGNORECASE | re.DOTALL)
    if match:
        content = match.group(1).strip()
        return content if content else None
    return None


def load_json(json_path: Path) -> Any:
    """Load JSON file."""
    with open(json_path, "r") as f:
        return json.load(f)


def save_json(data: Any, json_path: Path, indent: int = 2) -> None:
    """Save data to JSON file."""
    json_path.parent.mkdir(parents=True, exist_ok=True)
    with open(json_path, "w") as f:
        json.dump(data, f, indent=indent)


def load_text(text_path: Path) -> str:
    """Load text file."""
    with open(text_path, "r") as f:
        return f.read()


def save_text(text: str, text_path: Path) -> None:
    """Save text to file."""
    text_path.parent.mkdir(parents=True, exist_ok=True)
    with open(text_path, "w") as f:
        f.write(text)


def ensure_dir(dir_path: Path) -> Path:
    """Ensure directory exists."""
    dir_path.mkdir(parents=True, exist_ok=True)
    return dir_path


def find_files_by_pattern(
    directory: Path, 
    pattern: str, 
    recursive: bool = True
) -> List[Path]:
    """Find files matching a pattern."""
    if recursive:
        return list(directory.rglob(pattern))
    return list(directory.glob(pattern))


def format_timestamp(timestamp: float) -> str:
    """Format timestamp for filenames."""
    from datetime import datetime
    dt = datetime.fromtimestamp(timestamp)
    return dt.strftime("%Y%m%d_%H%M%S")