#!/usr/bin/env python3
"""Monitor a fine-tuning job's progress."""

import argparse
import os
import time
from datetime import datetime

from openai import OpenAI


def format_timestamp(timestamp: int) -> str:
    """Format Unix timestamp to human-readable string."""
    return datetime.fromtimestamp(timestamp).strftime("%Y-%m-%d %H:%M:%S")


def monitor_job(client: OpenAI, job_id: str, poll_interval: int = 30) -> None:
    """Monitor a fine-tuning job until completion."""
    print(f"Monitoring job: {job_id}")
    print(f"Polling every {poll_interval} seconds...\n")
    
    last_status = None
    
    while True:
        try:
            job = client.fine_tuning.jobs.retrieve(job_id)
            
            # Print update if status changed
            if job.status != last_status:
                print(f"[{format_timestamp(int(time.time()))}] Status: {job.status}")
                
                if job.status == "running":
                    print(f"  Training started at: {format_timestamp(job.started_at)}")
                elif job.status == "succeeded":
                    print(f"  Training completed at: {format_timestamp(job.finished_at)}")
                    print(f"  Fine-tuned model: {job.fine_tuned_model}")
                    print("\nJob completed successfully!")
                    break
                elif job.status == "failed":
                    print(f"  Error: {job.error}")
                    print("\nJob failed!")
                    break
                elif job.status == "cancelled":
                    print("\nJob was cancelled!")
                    break
                
                last_status = job.status
            
            # Show training metrics if available
            if job.status == "running" and job.trained_tokens:
                print(f"  Trained tokens: {job.trained_tokens:,}")
            
            if job.status in ["succeeded", "failed", "cancelled"]:
                break
            
            time.sleep(poll_interval)
            
        except KeyboardInterrupt:
            print("\nMonitoring interrupted by user")
            break
        except Exception as e:
            print(f"Error retrieving job status: {e}")
            time.sleep(poll_interval)


def main():
    parser = argparse.ArgumentParser(
        description="Monitor an OpenAI fine-tuning job"
    )
    
    parser.add_argument(
        "job_id",
        help="Fine-tuning job ID to monitor"
    )
    
    parser.add_argument(
        "--poll-interval",
        type=int,
        default=30,
        help="Polling interval in seconds (default: 30)"
    )
    
    parser.add_argument(
        "--api-key",
        help="OpenAI API key (defaults to OPENAI_API_KEY env var)"
    )
    
    args = parser.parse_args()
    
    # Get API key
    api_key = args.api_key or os.environ.get("OPENAI_API_KEY")
    if not api_key:
        raise ValueError("Please provide --api-key or set OPENAI_API_KEY")
    
    # Initialize client and monitor
    client = OpenAI(api_key=api_key)
    monitor_job(client, args.job_id, args.poll_interval)


if __name__ == "__main__":
    main()