#!/usr/bin/env python3
"""Complete pipeline for preparing data and starting fine-tuning."""

import argparse
import os
from pathlib import Path
from typing import Optional

from openai import OpenAI

from .prepare_data import prepare_synthetic_docs
from .start_fine_tuning import validate_training_file, upload_training_file, start_fine_tuning


def run_pipeline(
    input_path: Path,
    model: str = "gpt-4.1-mini-2025-04-14",
    suffix: Optional[str] = None,
    n_docs: Optional[int] = None,
    epochs: Optional[int] = None,
    output_dir: Optional[Path] = None,
    skip_preparation: bool = False,
    skip_upload: bool = False,
    file_id: Optional[str] = None,
) -> str:
    """
    Run complete fine-tuning pipeline.
    
    Args:
        input_path: Path to synthetic documents or prepared training data
        model: Base model to fine-tune
        suffix: Suffix for fine-tuned model
        n_docs: Number of documents to use for training
        epochs: Number of training epochs
        output_dir: Directory to save intermediate files
        skip_preparation: Skip data preparation (assume input is already formatted)
        skip_upload: Skip file upload (use existing file_id)
        file_id: Existing file ID to use
        
    Returns:
        Fine-tuning job ID
    """
    # Set up output directory
    if output_dir is None:
        output_dir = Path("data/fine_tuning")
    output_dir.mkdir(parents=True, exist_ok=True)
    
    # Step 1: Prepare data (unless skipped)
    if skip_preparation:
        training_file = input_path
        print(f"Using pre-formatted training data: {training_file}")
    else:
        training_file = output_dir / "training_data.jsonl"
        print(f"Preparing synthetic documents from: {input_path}")
        prepare_synthetic_docs(
            input_path=input_path,
            output_path=training_file,
            n_docs=n_docs,
            shuffle=True
        )
    
    # Validate training file
    validate_training_file(training_file)
    
    # Initialize OpenAI client
    api_key = os.environ.get("OPENAI_API_KEY")
    if not api_key:
        raise ValueError("Please set OPENAI_API_KEY environment variable")
    client = OpenAI(api_key=api_key)
    
    # Step 2: Upload file (unless skipped)
    if file_id:
        training_file_id = file_id
        print(f"Using existing file ID: {training_file_id}")
    elif skip_upload:
        raise ValueError("Must provide --file-id when using --skip-upload")
    else:
        training_file_id = upload_training_file(client, training_file)
    
    # Step 3: Start fine-tuning
    hyperparameters = {}
    if epochs is not None:
        hyperparameters["n_epochs"] = epochs
    
    job_id = start_fine_tuning(
        client=client,
        training_file_id=training_file_id,
        model=model,
        suffix=suffix,
        hyperparameters=hyperparameters if hyperparameters else None,
    )
    
    return job_id


def main():
    parser = argparse.ArgumentParser(
        description="Complete pipeline for synthetic document fine-tuning"
    )
    
    # Required arguments
    parser.add_argument(
        "input_path",
        type=Path,
        help="Path to synthetic documents JSONL or prepared training data"
    )
    
    # Model configuration
    parser.add_argument(
        "--model",
        default="gpt-4.1-mini-2025-04-14",
        help="Base model to fine-tune"
    )
    
    parser.add_argument(
        "--suffix",
        help="Suffix for fine-tuned model name"
    )
    
    # Data configuration
    parser.add_argument(
        "--n-docs",
        type=int,
        help="Number of documents to use for training"
    )
    
    parser.add_argument(
        "--epochs",
        type=int,
        help="Number of training epochs"
    )
    
    parser.add_argument(
        "--output-dir",
        type=Path,
        help="Directory to save intermediate files"
    )
    
    # Pipeline control
    parser.add_argument(
        "--skip-preparation",
        action="store_true",
        help="Skip data preparation (input is already formatted)"
    )
    
    parser.add_argument(
        "--skip-upload",
        action="store_true",
        help="Skip file upload (use existing file ID)"
    )
    
    parser.add_argument(
        "--file-id",
        help="Existing file ID to use (with --skip-upload)"
    )
    
    args = parser.parse_args()
    
    # Validate input
    if not args.input_path.exists():
        raise FileNotFoundError(f"Input file not found: {args.input_path}")
    
    # Run pipeline
    job_id = run_pipeline(
        input_path=args.input_path,
        model=args.model,
        suffix=args.suffix,
        n_docs=args.n_docs,
        epochs=args.epochs,
        output_dir=args.output_dir,
        skip_preparation=args.skip_preparation,
        skip_upload=args.skip_upload,
        file_id=args.file_id,
    )
    
    print(f"\nFine-tuning pipeline completed!")
    print(f"Job ID: {job_id}")
    print(f"\nMonitor with: python -m fine_tuning.monitor_job {job_id}")


if __name__ == "__main__":
    main()