#!/usr/bin/env python3
"""Convert synthetic documents to fine-tuning format."""

import argparse
import json
import random
from pathlib import Path
from typing import List, Dict, Any, Literal, Optional


def load_documents(file_path: Path) -> List[Any]:
    """Load documents from JSONL file."""
    docs = []
    with open(file_path, 'r') as f:
        for line in f:
            if line.strip():
                doc = json.loads(line)
                docs.append(doc)
    return docs


def synth_docs_to_ft_format(
    synthdoc_path: Path,
    output_path: Path,
    formatting: Literal["oai_messages_doctag", "together_text", "oai_messages_qa_pair"] = "oai_messages_doctag",
    n_synth_docs: Optional[int] = None,
    save: bool = True,
    shuffle_data: bool = True,
) -> List[Dict[str, Any]]:
    """
    Convert synthetic documents to the format required for fine-tuning.
    
    Args:
        synthdoc_path: Path to synthetic documents JSONL file
        output_path: Path to save formatted output
        formatting: Output format type
        n_synth_docs: Number of documents to use (None for all)
        save: Whether to save output to file
        shuffle_data: Whether to shuffle data before processing
    
    Returns:
        List of formatted training examples
    """
    # Load documents
    docs = load_documents(synthdoc_path)
    print(f"Loaded {len(docs)} documents from {synthdoc_path}")
    
    # Format based on specified type
    if formatting == "oai_messages_doctag":
        # Extract just the content field for each doc
        all_data = [
            {
                "messages": [
                    {"role": "user", "content": "<DOCTAG>"},
                    {"role": "assistant", "content": doc.get("content", str(doc))},
                ]
            }
            for doc in docs
        ]
    elif formatting == "together_text":
        # Simple text format
        all_data = [
            {"text": doc.get("content", str(doc))} 
            for doc in docs
        ]
    elif formatting == "oai_messages_qa_pair":
        # Assumes docs have question/answer fields
        all_data = []
        for doc in docs:
            if isinstance(doc, dict) and "question" in doc and "answer" in doc:
                all_data.append({
                    "messages": [
                        {"role": "user", "content": doc["question"]},
                        {"role": "assistant", "content": doc["answer"]},
                    ]
                })
            else:
                print(f"Skipping doc without question/answer fields")
    else:
        raise ValueError(f"Invalid formatting: {formatting}")
    
    # Shuffle if requested
    if shuffle_data:
        random.shuffle(all_data)
    
    # Limit number of docs if specified
    if n_synth_docs is not None:
        all_data = all_data[:n_synth_docs]
    
    print(f"Formatted {len(all_data)} documents")
    
    # Save if requested
    if save:
        output_path.parent.mkdir(parents=True, exist_ok=True)
        with open(output_path, "w") as f:
            for doc in all_data:
                f.write(json.dumps(doc) + "\n")
        print(f"Saved to {output_path}")
    
    return all_data


def prepare_synthetic_docs(
    input_path: Path,
    output_path: Path,
    n_docs: Optional[int] = None,
    shuffle: bool = True,
    formatting: str = "oai_messages_doctag"
) -> Path:
    """
    Prepare synthetic documents for fine-tuning (for backwards compatibility).
    
    Args:
        input_path: Path to synthetic documents
        output_path: Path to save formatted data
        n_docs: Number of documents to use
        shuffle: Whether to shuffle data
        formatting: Output format type
        
    Returns:
        Path to the formatted training data
    """
    formatted_data = synth_docs_to_ft_format(
        synthdoc_path=input_path,
        output_path=output_path,
        formatting=formatting,
        n_synth_docs=n_docs,
        save=True,
        shuffle_data=shuffle
    )
    
    print(f"Prepared {len(formatted_data)} training examples")
    return output_path


def main():
    parser = argparse.ArgumentParser(
        description="Convert synthetic documents to fine-tuning format"
    )
    
    # Required arguments
    parser.add_argument(
        "input_path",
        type=Path,
        help="Path to synthetic documents JSONL file"
    )
    
    parser.add_argument(
        "output_path",
        type=Path,
        help="Path to save formatted output"
    )
    
    # Optional arguments
    parser.add_argument(
        "--format",
        choices=["oai_messages_doctag", "together_text", "oai_messages_qa_pair"],
        default="oai_messages_doctag",
        help="Output format (default: oai_messages_doctag)"
    )
    
    parser.add_argument(
        "--n-docs",
        type=int,
        help="Number of documents to use (default: all)"
    )
    
    parser.add_argument(
        "--no-shuffle",
        action="store_true",
        help="Don't shuffle data"
    )
    
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Don't save output, just show stats"
    )
    
    args = parser.parse_args()
    
    # Validate input file exists
    if not args.input_path.exists():
        raise FileNotFoundError(f"Input file not found: {args.input_path}")
    
    # Convert documents
    formatted_data = synth_docs_to_ft_format(
        synthdoc_path=args.input_path,
        output_path=args.output_path,
        formatting=args.format,
        n_synth_docs=args.n_docs,
        save=not args.dry_run,
        shuffle_data=not args.no_shuffle,
    )
    
    # Show sample if dry run
    if args.dry_run and formatted_data:
        print("\nSample output (first item):")
        print(json.dumps(formatted_data[0], indent=2))


if __name__ == "__main__":
    main()