#!/usr/bin/env python3
"""Lightweight script to start an OpenAI fine-tuning job."""

import argparse
import json
import os
from pathlib import Path
from typing import Optional

from openai import OpenAI


def validate_training_file(file_path: Path) -> None:
    """Validate that the training file exists and has proper format."""
    if not file_path.exists():
        raise FileNotFoundError(f"Training file not found: {file_path}")

    # Basic validation - check it's valid JSONL
    with open(file_path, "r") as f:
        for i, line in enumerate(f):
            try:
                data = json.loads(line)
                if "messages" not in data:
                    raise ValueError(f"Line {i + 1}: Missing 'messages' field")
            except json.JSONDecodeError as e:
                raise ValueError(f"Line {i + 1}: Invalid JSON - {e}")


def upload_training_file(client: OpenAI, file_path: Path) -> str:
    """Upload training file and return file ID."""
    print(f"Uploading training file: {file_path}")

    with open(file_path, "rb") as f:
        response = client.files.create(file=f, purpose="fine-tune")

    file_id = response.id
    print(f"File uploaded successfully. ID: {file_id}")
    return file_id


def start_fine_tuning(
    client: OpenAI,
    training_file_id: str,
    model: str = "gpt-4.1-mini-2025-04-14",
    suffix: Optional[str] = None,
    hyperparameters: Optional[dict] = None,
    validation_file_id: Optional[str] = None,
) -> str:
    """Start fine-tuning job and return job ID."""
    print(f"Starting fine-tuning job on model: {model}")

    # Build request parameters
    params = {
        "training_file": training_file_id,
        "model": model,
    }

    if suffix:
        params["suffix"] = suffix

    if validation_file_id:
        params["validation_file"] = validation_file_id

    if hyperparameters:
        params["hyperparameters"] = hyperparameters

    # Create fine-tuning job
    job = client.fine_tuning.jobs.create(**params)

    print("Fine-tuning job created successfully!")
    print(f"Job ID: {job.id}")
    print(f"Status: {job.status}")

    return job.id


def main():
    parser = argparse.ArgumentParser(description="Start an OpenAI fine-tuning job")

    # Required arguments
    parser.add_argument("training_file", type=Path, help="Path to JSONL training file")

    # Optional arguments
    parser.add_argument(
        "--model",
        default="gpt-4.1-mini-2025-04-14",
        help="Base model to fine-tune (default: gpt-4.1-mini-2025-04-14)",
    )

    parser.add_argument(
        "--suffix", help="Suffix to add to fine-tuned model name (up to 64 chars)"
    )

    parser.add_argument(
        "--validation-file", type=Path, help="Path to optional JSONL validation file"
    )

    parser.add_argument("--epochs", type=int, help="Number of epochs to train for")

    parser.add_argument("--batch-size", type=int, help="Batch size for training")

    parser.add_argument(
        "--learning-rate-multiplier", type=float, help="Learning rate multiplier"
    )

    parser.add_argument(
        "--api-key", help="OpenAI API key (defaults to OPENAI_API_KEY env var)"
    )

    parser.add_argument(
        "--no-upload",
        action="store_true",
        help="Skip file upload (assume file already uploaded)",
    )

    parser.add_argument("--file-id", help="Use existing file ID instead of uploading")

    args = parser.parse_args()

    # Get API key
    api_key = args.api_key or os.environ.get("OPENAI_API_KEY")
    if not api_key:
        raise ValueError("Please provide --api-key or set OPENAI_API_KEY")

    # Initialize client
    client = OpenAI(api_key=api_key)

    # Handle file upload
    if args.file_id:
        training_file_id = args.file_id
        print(f"Using existing file ID: {training_file_id}")
    else:
        # Validate training file
        validate_training_file(args.training_file)

        # Upload training file
        training_file_id = upload_training_file(client, args.training_file)

    # Upload validation file if provided
    validation_file_id = None
    if args.validation_file:
        validate_training_file(args.validation_file)
        validation_file_id = upload_training_file(client, args.validation_file)

    # Build hyperparameters
    hyperparameters = {}
    if args.epochs is not None:
        hyperparameters["n_epochs"] = args.epochs
    if args.batch_size is not None:
        hyperparameters["batch_size"] = args.batch_size
    if args.learning_rate_multiplier is not None:
        hyperparameters["learning_rate_multiplier"] = args.learning_rate_multiplier

    # Start fine-tuning
    job_id = start_fine_tuning(
        client=client,
        training_file_id=training_file_id,
        model=args.model,
        suffix=args.suffix,
        hyperparameters=hyperparameters if hyperparameters else None,
        validation_file_id=validation_file_id,
    )

    print("\nTo monitor your fine-tuning job:")
    print(f"  python -m fine_tuning.monitor_job {job_id}")
    print("\nTo cancel the job:")
    print(f"  python -m fine_tuning.cancel_job {job_id}")


if __name__ == "__main__":
    main()
