#!/usr/bin/env python3
"""Utilities for managing OpenAI fine-tuning resources."""

import argparse
import os
from datetime import datetime
from typing import Optional, List, Dict, Any

from openai import OpenAI


def get_client(api_key: Optional[str] = None) -> OpenAI:
    """Get OpenAI client with API key."""
    api_key = api_key or os.environ.get("OPENAI_API_KEY")
    if not api_key:
        raise ValueError("Please provide API key or set OPENAI_API_KEY")
    return OpenAI(api_key=api_key)


def format_timestamp(timestamp: int) -> str:
    """Format Unix timestamp to human-readable string."""
    return datetime.fromtimestamp(timestamp).strftime("%Y-%m-%d %H:%M:%S")


def format_bytes(num_bytes: int) -> str:
    """Format bytes to human-readable string."""
    for unit in ["B", "KB", "MB", "GB"]:
        if num_bytes < 1024.0:
            return f"{num_bytes:.1f} {unit}"
        num_bytes /= 1024.0
    return f"{num_bytes:.1f} TB"


def list_files(
    client: OpenAI, 
    purpose: Optional[str] = None,
    limit: int = 100
) -> List[Dict[str, Any]]:
    """List uploaded files."""
    files = client.files.list(limit=limit)
    
    file_list = []
    for file in files.data:
        if purpose and file.purpose != purpose:
            continue
        
        file_info = {
            "id": file.id,
            "filename": file.filename,
            "purpose": file.purpose,
            "bytes": file.bytes,
            "created_at": file.created_at,
            "status": file.status,
        }
        file_list.append(file_info)
    
    return file_list


def list_fine_tuning_jobs(
    client: OpenAI,
    limit: int = 20,
    status: Optional[str] = None
) -> List[Dict[str, Any]]:
    """List fine-tuning jobs."""
    jobs = client.fine_tuning.jobs.list(limit=limit)
    
    job_list = []
    for job in jobs.data:
        if status and job.status != status:
            continue
        
        job_info = {
            "id": job.id,
            "model": job.model,
            "status": job.status,
            "created_at": job.created_at,
            "finished_at": job.finished_at,
            "fine_tuned_model": job.fine_tuned_model,
            "training_file": job.training_file,
            "error": job.error,
        }
        job_list.append(job_info)
    
    return job_list


def get_job_details(client: OpenAI, job_id: str) -> Dict[str, Any]:
    """Get detailed information about a fine-tuning job."""
    job = client.fine_tuning.jobs.retrieve(job_id)
    
    details = {
        "id": job.id,
        "model": job.model,
        "status": job.status,
        "created_at": format_timestamp(job.created_at),
        "started_at": format_timestamp(job.started_at) if job.started_at else None,
        "finished_at": format_timestamp(job.finished_at) if job.finished_at else None,
        "fine_tuned_model": job.fine_tuned_model,
        "training_file": job.training_file,
        "validation_file": job.validation_file,
        "hyperparameters": job.hyperparameters,
        "trained_tokens": job.trained_tokens,
        "error": job.error,
    }
    
    return details


def cancel_job(client: OpenAI, job_id: str) -> bool:
    """Cancel a fine-tuning job."""
    try:
        client.fine_tuning.jobs.cancel(job_id)
        return True
    except Exception as e:
        print(f"Error cancelling job: {e}")
        return False


def delete_file(client: OpenAI, file_id: str) -> bool:
    """Delete an uploaded file."""
    try:
        client.files.delete(file_id)
        return True
    except Exception as e:
        print(f"Error deleting file: {e}")
        return False


def delete_model(client: OpenAI, model_id: str) -> bool:
    """Delete a fine-tuned model."""
    try:
        client.models.delete(model_id)
        return True
    except Exception as e:
        print(f"Error deleting model: {e}")
        return False


def test_model(
    client: OpenAI,
    model: str,
    prompt: str,
    max_tokens: int = 500,
    temperature: float = 0.7,
    system_message: Optional[str] = None,
) -> Dict[str, Any]:
    """Test a model with a single prompt."""
    messages = []
    
    if system_message:
        messages.append({"role": "system", "content": system_message})
    
    messages.append({"role": "user", "content": prompt})
    
    response = client.chat.completions.create(
        model=model,
        messages=messages,
        max_tokens=max_tokens,
        temperature=temperature,
    )
    
    return {
        "prompt": prompt,
        "response": response.choices[0].message.content,
        "usage": response.usage.dict() if response.usage else None,
        "finish_reason": response.choices[0].finish_reason,
    }


def main():
    parser = argparse.ArgumentParser(
        description="Utilities for managing OpenAI fine-tuning resources"
    )
    
    subparsers = parser.add_subparsers(dest="command", help="Command to run")
    
    # List files command
    files_parser = subparsers.add_parser("files", help="List uploaded files")
    files_parser.add_argument(
        "--purpose",
        choices=["fine-tune", "fine-tune-results", "assistants", "assistants_output"],
        help="Filter by file purpose"
    )
    files_parser.add_argument(
        "--limit",
        type=int,
        default=100,
        help="Maximum number of files to list"
    )
    
    # List jobs command
    jobs_parser = subparsers.add_parser("jobs", help="List fine-tuning jobs")
    jobs_parser.add_argument(
        "--status",
        choices=["validating_files", "queued", "running", "succeeded", "failed", "cancelled"],
        help="Filter by job status"
    )
    jobs_parser.add_argument(
        "--limit",
        type=int,
        default=20,
        help="Maximum number of jobs to list"
    )
    
    # Job details command
    details_parser = subparsers.add_parser("details", help="Get job details")
    details_parser.add_argument("job_id", help="Fine-tuning job ID")
    
    # Cancel job command
    cancel_parser = subparsers.add_parser("cancel", help="Cancel a fine-tuning job")
    cancel_parser.add_argument("job_id", help="Fine-tuning job ID")
    
    # Delete file command
    delete_file_parser = subparsers.add_parser("delete-file", help="Delete an uploaded file")
    delete_file_parser.add_argument("file_id", help="File ID to delete")
    
    # Delete model command
    delete_model_parser = subparsers.add_parser("delete-model", help="Delete a fine-tuned model")
    delete_model_parser.add_argument("model_id", help="Model ID to delete")
    
    # List models command
    models_parser = subparsers.add_parser("models", help="List fine-tuned models")
    
    # Test model command
    test_parser = subparsers.add_parser("test", help="Test a model")
    test_parser.add_argument("model", help="Model ID to test")
    test_parser.add_argument("--prompt", default="<DOCTAG>", help="Prompt to test (default: <DOCTAG>)")
    test_parser.add_argument("--max-tokens", type=int, default=500, help="Max tokens to generate")
    test_parser.add_argument("--temperature", type=float, default=0.7, help="Temperature for generation")
    test_parser.add_argument("--system", help="System message")
    test_parser.add_argument("--num-samples", type=int, default=1, help="Number of samples to generate")
    
    # Global arguments
    parser.add_argument(
        "--api-key",
        help="OpenAI API key (defaults to OPENAI_API_KEY env var)"
    )
    
    args = parser.parse_args()
    
    if not args.command:
        parser.print_help()
        return
    
    # Get client
    client = get_client(args.api_key)
    
    # Execute command
    if args.command == "files":
        files = list_files(client, purpose=args.purpose, limit=args.limit)
        
        if not files:
            print("No files found")
            return
        
        print(f"\nFound {len(files)} files:\n")
        print(f"{'ID':<30} {'Filename':<40} {'Size':<12} {'Purpose':<15} {'Created'}")
        print("-" * 120)
        
        for file in files:
            print(
                f"{file['id']:<30} "
                f"{file['filename']:<40} "
                f"{format_bytes(file['bytes']):<12} "
                f"{file['purpose']:<15} "
                f"{format_timestamp(file['created_at'])}"
            )
    
    elif args.command == "jobs":
        jobs = list_fine_tuning_jobs(client, limit=args.limit, status=args.status)
        
        if not jobs:
            print("No jobs found")
            return
        
        print(f"\nFound {len(jobs)} jobs:\n")
        print(f"{'Job ID':<30} {'Model':<25} {'Status':<15} {'Created'}")
        print("-" * 100)
        
        for job in jobs:
            print(
                f"{job['id']:<30} "
                f"{job['model']:<25} "
                f"{job['status']:<15} "
                f"{format_timestamp(job['created_at'])}"
            )
            
            if job['fine_tuned_model']:
                print(f"  → Fine-tuned model: {job['fine_tuned_model']}")
            
            if job['error']:
                print(f"  → Error: {job['error']}")
    
    elif args.command == "details":
        details = get_job_details(client, args.job_id)
        
        print(f"\nJob Details for {details['id']}:")
        print("-" * 50)
        
        for key, value in details.items():
            if value is not None and key != "hyperparameters":
                print(f"{key:<20}: {value}")
        
        if details['hyperparameters']:
            print("\nHyperparameters:")
            for key, value in details['hyperparameters'].items():
                print(f"  {key:<20}: {value}")
    
    elif args.command == "cancel":
        if cancel_job(client, args.job_id):
            print(f"Successfully cancelled job: {args.job_id}")
        else:
            print(f"Failed to cancel job: {args.job_id}")
    
    elif args.command == "delete-file":
        if delete_file(client, args.file_id):
            print(f"Successfully deleted file: {args.file_id}")
        else:
            print(f"Failed to delete file: {args.file_id}")
    
    elif args.command == "delete-model":
        if delete_model(client, args.model_id):
            print(f"Successfully deleted model: {args.model_id}")
        else:
            print(f"Failed to delete model: {args.model_id}")
    
    elif args.command == "models":
        # List all models and filter for fine-tuned ones
        models = client.models.list()
        fine_tuned = [m for m in models.data if m.id.startswith("ft:")]
        
        if not fine_tuned:
            print("No fine-tuned models found")
            return
        
        print(f"\nFound {len(fine_tuned)} fine-tuned models:\n")
        print(f"{'Model ID':<60} {'Base Model':<25} {'Created'}")
        print("-" * 110)
        
        for model in fine_tuned:
            # Extract base model from ID (format: ft:model:org:suffix:id)
            parts = model.id.split(":")
            base_model = parts[1] if len(parts) > 1 else "unknown"
            
            print(
                f"{model.id:<60} "
                f"{base_model:<25} "
                f"{format_timestamp(model.created)}"
            )
    
    elif args.command == "test":
        # Test model
        print(f"\nTesting model: {args.model}")
        print(f"Prompt: {args.prompt}")
        print("-" * 80)
        
        for i in range(args.num_samples):
            if args.num_samples > 1:
                print(f"\nSample {i + 1}:")
            
            result = test_model(
                client=client,
                model=args.model,
                prompt=args.prompt,
                max_tokens=args.max_tokens,
                temperature=args.temperature,
                system_message=args.system,
            )
            
            print(f"\n{result['response']}")
            
            if result['usage'] and args.num_samples == 1:
                print(f"\nTokens - Prompt: {result['usage']['prompt_tokens']}, "
                      f"Completion: {result['usage']['completion_tokens']}, "
                      f"Total: {result['usage']['total_tokens']}")


if __name__ == "__main__":
    main()