#!/bin/bash

# Generalized script to run any evaluation pipeline
# Usage: 
#   ./run_evaluation.sh <evaluation> --config <config_file.yaml> [additional args]
#   ./run_evaluation.sh <evaluation> [command line args]
#
# Examples:
#   ./run_evaluation.sh alignment_faking --config configs/alignment_faking/default.yaml
#   ./run_evaluation.sh cot_faithfulness --config configs/cot_faithfulness/gpqa_test.yaml
#   ./run_evaluation.sh alignment_faking --model claude-3.5-sonnet-scratchpad-reveal-q --num-prompts 50

set -e  # Exit on error

# Function to print usage
usage() {
    echo "Usage:"
    echo "  $0 <evaluation> --config <config_file.yaml> [additional args]"
    echo "  $0 <evaluation> [command line args]"
    echo ""
    echo "Available evaluations:"
    echo "  alignment_faking     - Alignment faking evaluation"
    echo "  cot_faithfulness     - Chain of Thought faithfulness evaluation"
    echo "  model_written_evals  - Model-written evaluations from Anthropic paper"
    echo "  replot_mwe           - Regenerate plots from model-written eval results"
    echo "  llmcompare_eval      - LLMCompare flexible question-based evaluation"
    echo "  game_strategy        - Game strategy preservation evaluation"
    echo "  modification_reasoning - Modification reasoning detection evaluation"
    echo "  hint_templates       - Hint template comparison evaluation"
    echo ""
    echo "Examples:"
    echo "  $0 alignment_faking --config configs/alignment_faking/default.yaml"
    echo "  $0 cot_faithfulness --config configs/cot_faithfulness/gpqa_test.yaml"
    echo "  $0 model_written_evals --config configs/model_written_evals/default.yaml"
    echo "  $0 alignment_faking --model claude-3.5-sonnet-scratchpad-reveal-q --num-prompts 50"
    echo "  $0 model_written_evals --model claude-3-opus --categories persona --num-examples 50"
    echo "  $0 replot_mwe logs/model_written_evals/results_claude-3-opus_20240315_143022.json"
    echo "  $0 llmcompare_eval --config configs/llmcompare_eval/example_config.yaml"
    echo "  $0 game_strategy --config configs/game_strategy/default.yaml"
    echo "  $0 modification_reasoning --config configs/modification_reasoning/default.yaml"
    echo "  $0 hint_templates --config configs/hint_templates/gpqa_test.yaml"
    echo ""
    echo "Use --help after the evaluation name to see evaluation-specific options:"
    echo "  $0 alignment_faking --help"
    echo "  $0 cot_faithfulness --help"
    echo "  $0 model_written_evals --help"
    echo "  $0 llmcompare_eval --help"
    echo "  $0 game_strategy --help"
    echo "  $0 modification_reasoning --help"
    echo "  $0 hint_templates --help"
    exit 1
}

# Check arguments
if [ $# -lt 1 ]; then
    usage
fi

EVALUATION="$1"
shift  # Remove evaluation name from arguments

# Map evaluation names to Python module paths
case "$EVALUATION" in
    alignment_faking|af)
        MODULE="evaluations.alignment_faking.run_pipeline"
        ;;
    cot_faithfulness|cot)
        MODULE="evaluations.cot_faithfulness.run_pipeline"
        ;;
    model_written_evals|mwe)
        MODULE="evaluations.model_written_evals.run_pipeline"
        ;;
    replot_mwe)
        MODULE="evaluations.model_written_evals.replot_results"
        ;;
    llmcompare_eval|llmcompare)
        MODULE="evaluations.llmcompare_eval.main"
        ;;
    game_strategy|gs)
        MODULE="evaluations.game_strategy.run_pipeline"
        ;;
    modification_reasoning|mr)
        MODULE="evaluations.modification_reasoning.run_pipeline"
        ;;
    hint_templates|ht)
        MODULE="evaluations.hint_templates.run_pipeline"
        ;;
    *)
        echo "Error: Unknown evaluation '$EVALUATION'"
        echo ""
        usage
        ;;
esac

# Run the evaluation with all remaining arguments
echo "=========================================="
echo "Running $EVALUATION evaluation"
echo "=========================================="
echo ""

uv run python -m "$MODULE" "$@"