#!/usr/bin/env python3
"""
Download the full GPQA dataset.

This script downloads all splits of the GPQA dataset and saves them locally.
"""

import json
from pathlib import Path

from datasets import load_dataset


def download_gpqa():
    """Download the full GPQA dataset."""
    print("Downloading GPQA dataset...")

    # Load the full dataset (all splits)
    dataset = load_dataset("Idavidrein/gpqa", "gpqa_main")

    # Create data directory
    data_dir = Path("data")
    data_dir.mkdir(exist_ok=True)

    # Save each split
    for split_name, split_data in dataset.items():
        print(f"Processing {split_name} split: {len(split_data)} questions")

        # Save as dataset format
        split_dir = data_dir / f"gpqa_{split_name}"
        split_data.save_to_disk(str(split_dir))

        # Also save as JSON for easier inspection
        json_file = data_dir / f"gpqa_{split_name}.json"
        with open(json_file, "w") as f:
            json.dump(list(split_data), f, indent=2)

        print(f"Saved {split_name} to {split_dir} and {json_file}")

    print(f"\nDownload complete! Total splits: {len(dataset)}")
    return dataset


def main():
    dataset = download_gpqa()

    # Print summary
    print("\nDataset Summary:")
    for split_name, split_data in dataset.items():
        print(f"  {split_name}: {len(split_data)} questions")

    # Show sample question structure
    # if "train" in dataset:
    #     sample = dataset["train"][0]
    #     print("\nSample question structure:")
    #     for key, value in sample.items():
    #         print(
    #             f"  {key}: {str(value)[:100]}{'...' if len(str(value)) > 100 else ''}"
    #         )


if __name__ == "__main__":
    main()
