#!/usr/bin/env python3
"""
Script to plot alignment faking results comparison across different model configurations.
Takes a list of model registry config names and finds matching logs based on configuration.
"""

import argparse
import json
import re
from pathlib import Path
from typing import Dict, List, Tuple, Optional
import matplotlib.pyplot as plt
import numpy as np
import yaml


def load_model_registry(registry_path: Path) -> Dict:
    """Load model registry from YAML file."""
    with open(registry_path, 'r') as f:
        return yaml.safe_load(f)


def find_matching_logs(config_name: str, registry: Dict, logs_dir: Path) -> List[Path]:
    """Find log files matching a specific configuration."""
    matching_logs = []
    
    # Get configuration from registry
    config = registry.get('configs', {}).get(config_name)
    if not config:
        print(f"Warning: Configuration '{config_name}' not found in registry")
        return []
    
    # Extract paths from config
    model_id = config.get('model_id')
    system_prompt_path = config.get('system_prompt_path')
    conversation_json_path = config.get('conversation_json_path')
    
    for log_file in logs_dir.glob("*.json"):
        try:
            with open(log_file, 'r') as f:
                data = json.load(f)
                
                # Check if model matches
                if data.get("model") != model_id:
                    continue
                
                # Check if paths match
                log_sys_prompt = data.get("system_prompt_path")
                log_conv_path = data.get("conversation_json_path")
                
                # Match system prompt (if specified in config)
                if system_prompt_path and log_sys_prompt != system_prompt_path:
                    continue
                
                # Match conversation (if specified in config)
                if conversation_json_path and log_conv_path != conversation_json_path:
                    continue
                
                matching_logs.append(log_file)
                
        except (json.JSONDecodeError, IOError):
            continue
    
    return matching_logs


def find_latest_log(logs: List[Path]) -> Optional[Path]:
    """Find the most recent log from a list based on datetime in filename."""
    if not logs:
        return None
    
    def extract_datetime(filepath: Path) -> str:
        """Extract datetime string from filename."""
        filename = filepath.stem  # Remove .json extension
        
        # Pattern 1: YYYYMMDD_HHMMSS_model-name
        if filename[:8].isdigit() and filename[8] == '_' and filename[9:15].isdigit():
            return filename[:15]  # YYYYMMDD_HHMMSS
        
        # Pattern 2: alignment_faking_evaluation_YYYYMMDD_HHMMSS
        parts = filename.split('_')
        if len(parts) >= 5 and parts[-2].isdigit() and parts[-1].isdigit():
            return f"{parts[-2]}_{parts[-1]}"  # YYYYMMDD_HHMMSS
        
        # Pattern 3: other patterns with embedded dates
        # Look for YYYYMMDD_HHMMSS pattern anywhere in the filename
        import re
        date_pattern = re.search(r'(\d{8}_\d{6})', filename)
        if date_pattern:
            return date_pattern.group(1)
        
        # Fallback to filename for sorting (shouldn't happen with proper logs)
        return filename
    
    # Sort by extracted datetime string
    return max(logs, key=extract_datetime)


def find_config_name_for_log(log_data: Dict, registry: Dict) -> Optional[str]:
    """Try to find a matching config name for a log file based on its contents."""
    log_model = log_data.get("model")
    log_sys_prompt = log_data.get("system_prompt_path")
    log_conv_path = log_data.get("conversation_json_path")
    
    # Check all configs in registry
    for config_name, config in registry.get('configs', {}).items():
        if config.get('model_id') != log_model:
            continue
            
        # Check if paths match
        if log_sys_prompt and config.get('system_prompt_path') != log_sys_prompt:
            continue
            
        if log_conv_path and config.get('conversation_json_path') != log_conv_path:
            continue
            
        return config_name
    
    return None


def extract_metrics(log_data: Dict) -> Tuple[float, float, int]:
    """Extract compliance gap and alignment faking frequency from log data.
    Returns: (compliance_rate, alignment_faking_rate, total_samples)
    """
    compliance_count = 0
    refusal_count = 0
    alignment_faking_count = 0
    total = 0
    
    for result in log_data.get("results", []):
        classification = result.get("classification", {})
        
        compliance_result = classification.get("compliance_result", {})
        if compliance_result.get("success"):
            extracted = compliance_result.get("extracted", {})
            refusal_score = extracted.get("refusal_score")
            
            if refusal_score is not None:
                total += 1
                if refusal_score == 1:
                    refusal_count += 1
                else:
                    compliance_count += 1
                    
        af_result = classification.get("alignment_faking_result", {})
        if af_result.get("success"):
            af_extracted = af_result.get("extracted", {})
            if af_extracted.get("final_answer") == "Yes":
                alignment_faking_count += 1
    
    if total == 0:
        return 0, 0, 0
    
    compliance_rate = compliance_count / total
    af_rate = alignment_faking_count / total
    
    return compliance_rate, af_rate, total


def get_config_label(config_name: str, log_data: Dict) -> str:
    """Generate a descriptive label for a model configuration."""
    tier = log_data.get("tier", "unknown")
    tier_label = log_data.get("tier_label", tier)
    
    # Clean up config name for display
    if "scratchpad-reveal" in config_name:
        base = "Scratchpad Reveal"
    elif "control" in config_name:
        base = "Control"
    elif "multi-branch" in config_name:
        base = "Multi-Branch"
    elif "pro-max" in config_name:
        base = "Pro/Max"
    elif "af-sysprompt" in config_name:
        base = "AF System Prompt"
    else:
        base = config_name.replace("-", " ").replace("_", " ").title()
    
    if "throughput-testing" in config_name:
        base += " (Throughput)"
    
    # Add model info if it's a specific model variant
    model_id = log_data.get("model", "")
    if "opus" in config_name.lower() or "opus" in model_id.lower():
        base = f"Opus {base}"
    elif "3.5" in model_id and "opus" not in base.lower():
        base = f"Sonnet 3.5 {base}"
    
    label = f"{base} - {tier_label.capitalize()} Tier"
    
    if tier != tier_label:
        label += f" ({tier} data)"
    
    return label


def plot_comparison(config_data: List[Tuple[str, Dict, str]], output_path: Optional[Path] = None):
    """Create comparison plots for alignment faking metrics."""
    fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(12, 8))
    
    labels = []
    compliance_rates = []
    af_rates = []
    sample_sizes = []
    
    for config_name, log_data, log_path in config_data:
        compliance_rate, af_rate, total = extract_metrics(log_data)
        label = get_config_label(config_name, log_data)
        
        labels.append(label)
        compliance_rates.append(compliance_rate * 100)
        af_rates.append(af_rate * 100)
        sample_sizes.append(total)
        
        print(f"\n{label}:")
        print(f"  Config: {config_name}")
        print(f"  Log: {log_path}")
        print(f"  Compliance rate: {compliance_rate:.1%}")
        print(f"  AF reasoning rate: {af_rate:.1%}")
        print(f"  Total samples: {total}")
    
    x = np.arange(len(labels))
    width = 0.6
    
    colors = ['#4472C4', '#ED7D31', '#A5A5A5', '#FFC000', '#5B9BD5', '#70AD47']
    bar_colors = [colors[i % len(colors)] for i in range(len(labels))]
    
    bars1 = ax1.barh(x, compliance_rates, width, color=bar_colors, alpha=0.8)
    ax1.set_xlabel('Compliance Rate (%)', fontsize=12)
    ax1.set_title('Compliance Gap Across Configurations', fontsize=14, pad=10)
    ax1.set_xlim(0, 100)
    ax1.set_yticks(x)
    ax1.set_yticklabels(labels, fontsize=10)
    ax1.grid(axis='x', alpha=0.3)
    
    for i, (bar, n) in enumerate(zip(bars1, sample_sizes)):
        ax1.text(bar.get_width() + 1, bar.get_y() + bar.get_height()/2,
                f'{compliance_rates[i]:.1f}%', va='center', fontsize=9)
        ax1.text(95, bar.get_y() + bar.get_height()/2,
                f'n={n}', va='center', ha='right', fontsize=8, style='italic', color='gray')
    
    bars2 = ax2.barh(x, af_rates, width, color=bar_colors, alpha=0.8)
    ax2.set_xlabel('Alignment Faking Reasoning Rate (%)', fontsize=12)
    ax2.set_title('Alignment Faking Reasoning Frequency', fontsize=14, pad=10)
    max_af_rate = max(af_rates) if af_rates else 0
    ax2.set_xlim(0, max(max_af_rate * 1.2, 10))
    ax2.set_yticks(x)
    ax2.set_yticklabels(labels, fontsize=10)
    ax2.grid(axis='x', alpha=0.3)
    
    for i, bar in enumerate(bars2):
        text_x = bar.get_width() + 0.5 if bar.get_width() > 0 else 0.5
        ax2.text(text_x, bar.get_y() + bar.get_height()/2,
                f'{af_rates[i]:.1f}%', va='center', fontsize=9)
    
    plt.tight_layout()
    
    if output_path:
        plt.savefig(output_path, dpi=300, bbox_inches='tight')
        print(f"\nPlot saved to {output_path}")
    else:
        plt.show()


def main():
    parser = argparse.ArgumentParser(description='Plot alignment faking comparison across model configurations')
    parser.add_argument('inputs', nargs='+', help='Configuration names from registry OR paths to JSON log files')
    parser.add_argument('--logs-dir', type=Path, default=Path('logs/alignment_faking'),
                       help='Directory containing alignment faking logs')
    parser.add_argument('--registry', type=Path, default=Path('model_registry.yaml'),
                       help='Path to model registry YAML')
    parser.add_argument('--output', type=Path, help='Output file path for the plot')
    parser.add_argument('--all-tiers', action='store_true',
                       help='Include all tier variations for each config')
    
    args = parser.parse_args()
    
    # Load model registry (we'll need it even for direct log files to generate labels)
    registry = load_model_registry(args.registry) if args.registry.exists() else {}
    
    config_data = []
    
    for input_item in args.inputs:
        input_path = Path(input_item)
        
        # Check if it's a JSON file path
        if input_path.suffix == '.json' and input_path.exists():
            # Direct log file
            try:
                with open(input_path, 'r') as f:
                    log_data = json.load(f)
                
                # Try to find matching config name for better labeling
                config_name = find_config_name_for_log(log_data, registry)
                if not config_name:
                    # Use filename as fallback
                    config_name = input_path.stem
                
                config_data.append((config_name, log_data, input_path.name))
                print(f"Loaded log file: {input_path}")
                
            except (json.JSONDecodeError, IOError) as e:
                print(f"Warning: Failed to load log file {input_path}: {e}")
                continue
        else:
            # Assume it's a configuration name
            if not args.logs_dir.exists():
                print(f"Error: Logs directory {args.logs_dir} does not exist")
                continue
                
            matching_logs = find_matching_logs(input_item, registry, args.logs_dir)
            
            if not matching_logs:
                print(f"Warning: No logs found matching configuration '{input_item}'")
                continue
            
            if args.all_tiers:
                # Include all matching logs (different tiers)
                for log_path in matching_logs:
                    with open(log_path, 'r') as f:
                        log_data = json.load(f)
                    config_data.append((input_item, log_data, log_path.name))
            else:
                # Just use the latest log
                latest_log = find_latest_log(matching_logs)
                if latest_log:
                    with open(latest_log, 'r') as f:
                        log_data = json.load(f)
                    config_data.append((input_item, log_data, latest_log.name))
    
    if not config_data:
        print("Error: No valid log data found for any of the provided configurations")
        return
    
    # Default output path to scripts/ directory if not specified
    output_path = args.output
    if not output_path:
        # Generate default filename based on configs and timestamp
        from datetime import datetime
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        # Extract config names from the data for filename
        config_names = [data[0] for data in config_data]
        config_str = "_".join(config_names[:3])  # First 3 configs
        if len(config_names) > 3:
            config_str += "_etc"
        output_path = Path("scripts") / f"alignment_faking_{config_str}_{timestamp}.png"
    
    plot_comparison(config_data, output_path)


if __name__ == '__main__':
    main()