import argparse
import json
from pathlib import Path

import matplotlib.pyplot as plt
import numpy as np
import yaml


def extract_metrics(log_data):
    results = [r for r in log_data.get("results", []) if r.get("success")]
    metrics = {"total": 0, "compliant": 0, "af_compliant": 0, "af_noncompliant": 0}

    for r in results:
        comp = r.get("classification", {}).get("compliance_result", {})
        af = r.get("classification", {}).get("alignment_faking_result", {})

        if not (comp.get("success") and af.get("success")):
            continue

        metrics["total"] += 1
        is_compliant = comp["extracted"]["refusal_score"] == 0
        is_af = af["extracted"]["final_answer"] == "Yes"

        if is_compliant:
            metrics["compliant"] += 1
            if is_af:
                metrics["af_compliant"] += 1
        elif is_af:
            metrics["af_noncompliant"] += 1

    comp_rate = metrics["compliant"] / metrics["total"] * 100 if metrics["total"] else 0
    af_comp_rate = (
        metrics["af_compliant"] / metrics["compliant"] * 100
        if metrics["compliant"]
        else 0
    )
    af_noncomp_rate = (
        metrics["af_noncompliant"] / (metrics["total"] - metrics["compliant"]) * 100
        if metrics["total"] > metrics["compliant"]
        else 0
    )

    return {
        "compliance_rate": comp_rate,
        "af_when_complying": af_comp_rate * comp_rate / 100,
        "af_when_not_complying": af_noncomp_rate * (100 - comp_rate) / 100,
        "af_rate": (metrics["af_compliant"] + metrics["af_noncompliant"])
        / metrics["total"]
        * 100
        if metrics["total"]
        else 0,
    }


def process_setup(config, setup_name):
    setup = config["setups"][setup_name]
    log_pairs = []

    if "log_pairs" in setup:
        log_pairs = [(Path(p["free"]), Path(p["paid"])) for p in setup["log_pairs"]]
    elif "free_logs" in setup:
        log_pairs = list(
            zip(
                [Path(log) for log in setup["free_logs"]],
                [Path(log) for log in setup["paid_logs"]],
            )
        )
    else:
        log_pairs = [(Path(setup["free_log"]), Path(setup["paid_log"]))]

    metrics_list = []
    for free_path, paid_path in log_pairs:
        with open(free_path) as f:
            free_m = extract_metrics(json.load(f))
        with open(paid_path) as f:
            paid_m = extract_metrics(json.load(f))

        metrics_list.append(
            {
                "compliance_gap": free_m["compliance_rate"] - paid_m["compliance_rate"],
                "af_when_complying": free_m["af_when_complying"],
                "af_when_not_complying": free_m["af_when_not_complying"],
                "af_rate": free_m["af_rate"],
            }
        )

    if len(metrics_list) > 1:
        return {
            k: {
                "mean": np.mean([m[k] for m in metrics_list]),
                "std": np.std([m[k] for m in metrics_list]),
            }
            for k in metrics_list[0]
        }
    else:
        return {k: {"mean": metrics_list[0][k], "std": 0} for k in metrics_list[0]}


def plot_stacked_bars(config, output_path=None):
    # Keep the same serif font settings
    plt.rcParams.update(
        {
            "font.family": "serif",
            "font.serif": ["Times New Roman", "Times", "DejaVu Serif", "serif"],
        }
    )

    # ---- collect metrics for each setup (unchanged) ----
    data = []
    for name in config["setups"]:
        metrics = process_setup(config, name)
        data.append(
            {
                "name": name,
                "compliance_gap": metrics["compliance_gap"]["mean"],
                "compliance_gap_err": metrics["compliance_gap"]["std"],
                "af_when_complying": metrics["af_when_complying"]["mean"],
                "af_when_complying_err": metrics["af_when_complying"]["std"],
                "af_when_not_complying": metrics["af_when_not_complying"]["mean"],
                "af_when_not_complying_err": metrics["af_when_not_complying"]["std"],
            }
        )
        print(f"\n{name}:")
        print(
            f"  Compliance gap: {metrics['compliance_gap']['mean']:.1f}% ± {metrics['compliance_gap']['std']:.1f}%"
        )
        print(
            f"  AF when complying: {metrics['af_when_complying']['mean']:.1f}% ± {metrics['af_when_complying']['std']:.1f}%"
        )
        print(
            f"  AF when not complying: {metrics['af_when_not_complying']['mean']:.1f}% ± {metrics['af_when_not_complying']['std']:.1f}%"
        )

    n = len(data)

    # ---- figure: 1 row x 2 columns (metrics as columns) ----
    # width identical to before (2 panels * 3.5), height scales with number of setups
    fig_height = 3.5  # keeps the per-row look consistent if n > 2
    fig, axes = plt.subplots(1, 2, figsize=(2 * 3.5, fig_height), sharey=True)
    plt.subplots_adjust(wspace=0)

    # vertical placement now depends on the number of setups
    bar_height = 0.075
    step = 0.08  # vertical spacing between rows; matches the original look for two rows
    # symmetric positions around 0, e.g. n=2 -> [0.04, -0.04]
    y_pos = [((n - 1) / 2 - i) * step for i in range(n)]
    error_kw = {"elinewidth": 0.75, "capsize": 2, "alpha": 0.75, "ecolor": "0.1"}

    # ---- Left column: Compliance Gap ----
    ax_cg = axes[0]
    for i, d in enumerate(data):
        ax_cg.barh(
            y_pos[i],
            d["compliance_gap"],
            bar_height,
            color="#87CEEB",
            label="Compliance Gap" if i == 0 else "",
            xerr=d["compliance_gap_err"],
            error_kw=error_kw,
        )

    # ---- Right column: Alignment Faking (stacked) ----
    ax_af = axes[1]
    for i, d in enumerate(data):
        ax_af.barh(
            y_pos[i],
            d["af_when_complying"],
            bar_height,
            color="#cc0000",
            alpha=0.8,
            label="AF when complying" if i == 0 else "",
            xerr=d["af_when_complying_err"],
            error_kw=error_kw,
        )
        ax_af.barh(
            y_pos[i],
            d["af_when_not_complying"],
            bar_height,
            left=d["af_when_complying"],
            color="#ff6b6b",
            alpha=0.8,
            label="AF when not complying" if i == 0 else "",
            xerr=d["af_when_not_complying_err"],
            error_kw=error_kw,
        )

    # ---- common formatting (kept visually identical) ----
    for j, ax in enumerate(axes):
        ax.set_xlim(0, 50)
        pad = 0.045  # matches original headroom: [-0.085, 0.085] when n == 2
        ax.set_ylim(min(y_pos) - pad, max(y_pos) + pad)
        ax.set_yticks(y_pos)
        ax.set_yticklabels([d["name"] for d in data])
        ax.grid(True, alpha=0.1, axis="x")
        ax.set_axisbelow(True)

        # column titles are now the metric names
        ax.set_title(
            "Compliance Gap (%)" if j == 0 else "Alignment Faking Reasoning (%)",
            fontsize=11,
            pad=10,
        )
        for spine in ["top", "right"] + (["left"] if j > 0 else []):
            ax.spines[spine].set_visible(False)
        if j > 0:
            ax.tick_params(left=False)

    # legend identical to before
    from matplotlib.patches import Rectangle

    colors = ["#87CEEB", "#cc0000", "#ff6b6b"]
    alphas = [0.8, 0.8, 1.0]
    handles = [Rectangle((0, 0), 1, 1, fc=c, alpha=a) for c, a in zip(colors, alphas)]
    fig.legend(
        handles=handles,
        labels=["Compliance Gap", "AF when complying", "AF when not complying"],
        loc="lower center",
        ncol=3,
        frameon=False,
        fontsize=11,
        bbox_to_anchor=(0.5, -0.15),
        handlelength=1,
        handleheight=1,
    )

    plt.tight_layout()
    if output_path:
        plt.savefig(output_path, dpi=300, bbox_inches="tight")
        print(f"\nPlot saved to: {output_path}")
    else:
        plt.show()


def plot_scatter(config, output_path=None):
    data = []
    for name in config["setups"]:
        metrics = process_setup(config, name)
        data.append(
            {
                "name": name,
                "compliance_gap": metrics["compliance_gap"]["mean"],
                "af_rate": metrics["af_rate"]["mean"],
            }
        )
        print(f"\n{name}:")
        print(f"  Compliance gap: {metrics['compliance_gap']['mean']:.1f}%")
        print(f"  AF rate (free tier): {metrics['af_rate']['mean']:.1f}%")

    _, ax = plt.subplots(figsize=(10, 8))

    markers = ["o", "s", "^", "D", "v", "*", "p", "h", "<", ">", "P", "X"]
    colors = [
        "#0081A7",
        "#F77F00",
        "#9D4EDD",
        "#06A77D",
        "#D62246",
        "#003049",
        "#F4A460",
        "#2E8B57",
        "#FF1493",
        "#4682B4",
        "#8B4513",
        "#DA70D6",
    ]

    for i, d in enumerate(data):
        ax.scatter(
            d["compliance_gap"],
            d["af_rate"],
            s=200,
            alpha=0.8,
            c=colors[i % len(colors)],
            marker=markers[i % len(markers)],
            label=d["name"],
            edgecolors="black",
            linewidth=0.5,
        )

    ax.set_xlabel("Compliance Gap (%)", fontsize=12)
    ax.set_ylabel("Alignment Faking Rate - Free Tier (%)", fontsize=12)
    ax.set_title("Alignment Faking vs Compliance Gap", fontsize=14, fontweight="bold")
    ax.set_xlim(0, 100)
    ax.set_ylim(0, 100)
    ax.grid(True, alpha=0.3)
    ax.legend(
        loc="upper center",
        frameon=True,
        fancybox=True,
        shadow=True,
        bbox_to_anchor=(0.5, -0.15),
        ncol=3,
        borderaxespad=0,
    )

    plt.tight_layout()
    if output_path:
        plt.savefig(output_path, dpi=300, bbox_inches="tight")
        print(f"\nPlot saved to: {output_path}")
    else:
        plt.show()


def main():
    parser = argparse.ArgumentParser(
        description="Plot compliance gap and alignment faking rates"
    )
    parser.add_argument(
        "--config",
        type=Path,
        required=True,
        help="YAML configuration file with log paths",
    )
    parser.add_argument("--output", type=Path, help="Output file path for the plot")
    parser.add_argument(
        "--scatter",
        action="store_true",
        help="Create scatter plot instead of bar chart",
    )
    parser.add_argument(
        "--stacked", action="store_true", help="Create stacked bar plot"
    )

    args = parser.parse_args()

    with open(args.config) as f:
        config = yaml.safe_load(f)

    if args.scatter:
        plot_scatter(config, args.output)
    else:
        plot_stacked_bars(config, args.output)


if __name__ == "__main__":
    main()
