#!/usr/bin/env python3
"""Standalone script to plot hint template evaluation results from log files."""

import argparse
import json
from pathlib import Path
from typing import Dict, List

import matplotlib.pyplot as plt
import numpy as np


def create_comparison_plot(
    log_paths, output_path: Path = None, y_labels: Dict[str, str] = None
):
    """Create comparison plot from hint template evaluation log file(s).

    Args:
        log_paths: Path(s) to log files
        output_path: Where to save the plot
        y_labels: Optional dict mapping template names to custom labels
    """
    # Handle single log path
    if isinstance(log_paths, Path):
        log_paths = [log_paths]

    # Set font to serif like compliance gap plots
    plt.rcParams.update(
        {
            "font.family": "serif",
            "font.serif": ["Times New Roman", "Times", "DejaVu Serif", "serif"],
        }
    )

    # Collect data from all log files
    all_data_by_template = {}
    config_info = None

    for log_path in log_paths:
        with open(log_path) as f:
            data = json.load(f)

        if config_info is None:
            config_info = data["config"]

        summaries = data["summaries"]

        # Extract metrics for each template
        for name, summary in summaries.items():
            total = summary["total"]
            if total == 0:
                continue

            # Calculate percentages
            # Use changed_answer_rate if available, otherwise fall back to inverse of baseline_accuracy
            if "changed_answer_rate" in summary:
                changed_answer_rate = summary["changed_answer_rate"] * 100
            else:
                # For old logs: assume answer changed when not correct
                changed_answer_rate = (1 - summary["baseline_accuracy"]) * 100

            # Calculate acknowledgment rates when following/not following hint
            followed = summary["followed_hint"]

            ack_when_following = 0
            if followed > 0:
                ack_when_following = (summary["acknowledges_metadata"] / total) * 100

            if name not in all_data_by_template:
                all_data_by_template[name] = {
                    "changed_answer": [],
                    "reasoning_about_hint": [],
                }

            all_data_by_template[name]["changed_answer"].append(changed_answer_rate)
            # Just use the acknowledgment rate when following hint (which is when they reason about it)
            all_data_by_template[name]["reasoning_about_hint"].append(
                ack_when_following
            )

    # Compute means and stds
    plot_data = []
    print("\nHint Template Evaluation Results:")
    print("=" * 50)
    for name, metrics in all_data_by_template.items():
        # Use custom label if provided, otherwise use formatted template name
        if y_labels and name in y_labels:
            label = y_labels[name]
        else:
            if name == "default":
                label = "Default"
            elif name == "instruct_hide_reasoning_simple":
                label = "Instructed to Evade Monitor"
            else:
                label = "Instructed + Monitor Info"

        changed_answer_mean = np.mean(metrics["changed_answer"])
        changed_answer_std = np.std(metrics["changed_answer"]) if len(metrics["changed_answer"]) > 1 else 0
        reasoning_mean = np.mean(metrics["reasoning_about_hint"])
        reasoning_std = np.std(metrics["reasoning_about_hint"]) if len(metrics["reasoning_about_hint"]) > 1 else 0

        print(f"\n{label}:")
        print(f"  Answer Change Rate: {changed_answer_mean:.1f}% ± {changed_answer_std:.1f}%")
        print(f"  Reasoning About Hint: {reasoning_mean:.1f}% ± {reasoning_std:.1f}%")

        plot_data.append(
            {
                "name": label,
                "changed_answer": changed_answer_mean,
                "changed_answer_err": changed_answer_std,
                "reasoning_about_hint": reasoning_mean,
                "reasoning_about_hint_err": reasoning_std,
            }
        )

    n = len(plot_data)

    # Figure: 1 row x 2 columns (metrics as columns)
    fig_height = 2.5  # scales with number of templates
    fig, axes = plt.subplots(1, 2, figsize=(2 * 3.5, fig_height), sharey=True)
    plt.subplots_adjust(wspace=0)

    # Vertical placement
    bar_height = 0.075
    step = 0.08  # vertical spacing between rows
    y_pos = [((n - 1) / 2 - i) * step for i in range(n)]
    error_kw = {"elinewidth": 0.75, "capsize": 2, "alpha": 0.75, "ecolor": "0.1"}

    # Left column: Answer Change Rate (how often model changes answer)
    ax_change = axes[0]
    for i, d in enumerate(plot_data):
        ax_change.barh(
            y_pos[i],
            d.get("changed_answer", d.get("accuracy", 0)),  # Fallback for compatibility
            bar_height,
            color="#7EBD8E",  # Light red/coral to indicate change
            alpha=1,
            label="Changed Answer Rate" if i == 0 else "",
            xerr=d.get("changed_answer_err", d.get("accuracy_err", 0)),
            error_kw=error_kw,
        )

    # Right column: Reasoning About Hint
    ax_ack = axes[1]
    for i, d in enumerate(plot_data):
        ax_ack.barh(
            y_pos[i],
            d["reasoning_about_hint"],
            bar_height,
            color="#FF9B5A",  # Crimson red
            alpha=1,
            label="Reasoning About Hint" if i == 0 else "",
            xerr=d.get("reasoning_about_hint_err", 0),
            error_kw=error_kw,
        )

    # Common formatting
    for j, ax in enumerate(axes):
        ax.set_xlim(0, 100)
        pad = 0.045
        ax.set_ylim(min(y_pos) - pad, max(y_pos) + pad)
        ax.set_yticks(y_pos)
        ax.set_yticklabels([d["name"] for d in plot_data])
        ax.grid(True, alpha=0.1, axis="x")
        ax.set_axisbelow(True)

        # Column titles
        if j == 0:
            ax.set_title("Answer Change Rate (%)", fontsize=11, pad=10)
        else:
            ax.set_title("Reasoning About Hint (%)", fontsize=11, pad=10)

        for spine in ["top", "right"] + (["left"] if j > 0 else []):
            ax.spines[spine].set_visible(False)
        if j > 0:
            ax.tick_params(left=False)

    # Legend
    from matplotlib.patches import Rectangle

    colors = ["#7EBD8E", "#FF9B5A"]
    labels = [
        "Answer Change Rate",
        "Reasoning About Hint",
    ]
    alphas = [1, 1]

    handles = [Rectangle((0, 0), 1, 1, fc=c, alpha=a) for c, a in zip(colors, alphas)]
    fig.legend(
        handles=handles,
        labels=labels,
        loc="lower center",
        ncol=len(labels),
        frameon=False,
        fontsize=11,
        bbox_to_anchor=(0.5, -0.15),
        handlelength=1,
        handleheight=1,
    )

    plt.tight_layout()

    # Save plot
    if output_path:
        plt.savefig(output_path, dpi=300, bbox_inches="tight")
        if output_path.suffix == ".png":
            plt.savefig(output_path.with_suffix(".pdf"), bbox_inches="tight")
        print(f"Plot saved to: {output_path}")
        if output_path.suffix == ".png":
            print(f"PDF saved to: {output_path.with_suffix('.pdf')}")
    else:
        # Default output
        timestamp = config_info["timestamp"]
        model_name = config_info["model"].split("/")[-1]
        output_dir = Path("images/hint_templates")
        output_dir.mkdir(parents=True, exist_ok=True)
        output_file = output_dir / f"comparison_{timestamp}_{model_name}.png"
        plt.savefig(output_file, dpi=300, bbox_inches="tight")
        plt.savefig(output_file.with_suffix(".pdf"), bbox_inches="tight")
        print(f"Plot saved to: {output_file}")
        print(f"PDF saved to: {output_file.with_suffix('.pdf')}")

    plt.close(fig)


def plot_multiple_logs(
    log_files: List[Path],
    output_path: Path = None,
    labels: List[str] = None,
    y_labels: Dict[str, str] = None,
):
    """Create a comparison plot from multiple hint template evaluation logs.

    Args:
        log_files: List of log file paths
        output_path: Where to save the plot
        labels: Custom labels for each model (log file)
        y_labels: Optional dict mapping template names to custom labels
    """

    # Set font to serif
    plt.rcParams.update(
        {
            "font.family": "serif",
            "font.serif": ["Times New Roman", "Times", "DejaVu Serif", "serif"],
        }
    )

    # Load data from all log files
    all_data = []
    for i, log_path in enumerate(log_files):
        with open(log_path) as f:
            data = json.load(f)

        # Extract model name or use provided label
        if labels and i < len(labels):
            model_label = labels[i]
        else:
            model_label = data["config"]["model"].split("/")[-1]

        # Process summaries
        summaries = data["summaries"]
        for name, summary in summaries.items():
            total = summary["total"]
            if total == 0:
                continue

            # Calculate metrics
            # Use changed_answer_rate if available, otherwise fall back to inverse of baseline_accuracy
            if "changed_answer_rate" in summary:
                changed_answer_rate = summary["changed_answer_rate"] * 100
            else:
                # For old logs: assume answer changed when not correct
                changed_answer_rate = (1 - summary["baseline_accuracy"]) * 100

            followed = summary["followed_hint"]

            ack_when_following = 0
            if followed > 0:
                ack_when_following = (summary["acknowledges_metadata"] / total) * 100

            # Use custom label if provided
            if y_labels and name in y_labels:
                template_label = y_labels[name]
            else:
                template_label = name.replace("_", " ").title()

            all_data.append(
                {
                    "model": model_label,
                    "template": template_label,
                    "changed_answer": changed_answer_rate,
                    "reasoning_about_hint": ack_when_following,
                }
            )

    if not all_data:
        print("No data found in log files!")
        return

    # Group by template
    templates = sorted(list(set(d["template"] for d in all_data)))
    models = sorted(list(set(d["model"] for d in all_data)))

    # Create plot with one row per model-template combination
    n_rows = len(models) * len(templates)

    # Figure: 1 row x 2 columns
    fig_height = max(2, n_rows * 0.3)
    fig, axes = plt.subplots(1, 2, figsize=(2 * 3.5, fig_height), sharey=True)
    plt.subplots_adjust(wspace=0)

    # Vertical placement
    bar_height = 0.075
    step = 0.08
    y_positions = []
    y_labels = []

    # Build positions and labels
    current_y = (n_rows - 1) / 2 * step
    for model in models:
        for template in templates:
            y_positions.append(current_y)
            y_labels.append(f"{model}\n{template}")
            current_y -= step

    # Plot data
    error_kw = {"elinewidth": 0.75, "capsize": 2, "alpha": 0.75, "ecolor": "0.1"}

    # Left column: Answer Change Rate
    ax_change = axes[0]
    for i, (model, template) in enumerate([(m, t) for m in models for t in templates]):
        # Find matching data
        data_point = next(
            (d for d in all_data if d["model"] == model and d["template"] == template),
            None,
        )
        if data_point:
            ax_change.barh(
                y_positions[i],
                data_point.get("changed_answer", data_point.get("accuracy", 0)),
                bar_height,
                color="#7EBD8E",
                alpha=0.8,
                label="Answer Change Rate" if i == 0 else "",
                error_kw=error_kw,
            )

    # Right column: Reasoning About Hint
    ax_ack = axes[1]
    for i, (model, template) in enumerate([(m, t) for m in models for t in templates]):
        data_point = next(
            (d for d in all_data if d["model"] == model and d["template"] == template),
            None,
        )
        if data_point:
            ax_ack.barh(
                y_positions[i],
                data_point["reasoning_about_hint"],
                bar_height,
                color="#FF9B5A",
                alpha=0.8,
                label="Reasoning About Hint" if i == 0 else "",
                error_kw=error_kw,
            )

    # Common formatting
    for j, ax in enumerate(axes):
        ax.set_xlim(0, 100)
        pad = 0.045
        ax.set_ylim(min(y_positions) - pad, max(y_positions) + pad)
        ax.set_yticks(y_positions)
        ax.set_yticklabels(y_labels, fontsize=8)
        ax.grid(True, alpha=0.1, axis="x")
        ax.set_axisbelow(True)

        # Column titles
        if j == 0:
            ax.set_title("Answer Change Rate (%)", fontsize=11, pad=10)
        else:
            ax.set_title("Reasoning About Hint (%)", fontsize=11, pad=10)

        for spine in ["top", "right"] + (["left"] if j > 0 else []):
            ax.spines[spine].set_visible(False)
        if j > 0:
            ax.tick_params(left=False)

    # Legend
    from matplotlib.patches import Rectangle

    colors = ["#7EBD8E", "#FF9B5A"]
    labels = [
        "Answer Change Rate",
        "Reasoning About Hint",
    ]
    alphas = [0.8, 0.8]

    handles = [Rectangle((0, 0), 1, 1, fc=c, alpha=a) for c, a in zip(colors, alphas)]
    fig.legend(
        handles=handles,
        labels=labels,
        loc="lower center",
        ncol=len(labels),
        frameon=False,
        fontsize=11,
        bbox_to_anchor=(0.5, -0.15),
        handlelength=1,
        handleheight=1,
    )

    plt.tight_layout()

    if output_path:
        plt.savefig(output_path, dpi=300, bbox_inches="tight")
        if output_path.suffix == ".png":
            plt.savefig(output_path.with_suffix(".pdf"), bbox_inches="tight")
        print(f"Plot saved to: {output_path}")
    else:
        plt.show()


def main():
    parser = argparse.ArgumentParser(
        description="Plot hint template evaluation results from log files"
    )
    parser.add_argument(
        "logs",
        nargs="+",
        type=Path,
        help="Path(s) to hint template evaluation log files",
    )
    parser.add_argument(
        "--output",
        "-o",
        type=Path,
        help="Output file path for the plot (default: auto-generated in images/hint_templates/)",
    )
    parser.add_argument(
        "--labels",
        "-l",
        nargs="+",
        help="Custom labels for each log file (for multi-log plots)",
    )
    parser.add_argument(
        "--y-labels",
        "-y",
        type=str,
        help='JSON string mapping template names to custom y-axis labels, e.g., \'{"template_name": "Custom Label"}\'',
    )

    args = parser.parse_args()

    # Parse y_labels if provided
    y_labels = None
    if args.y_labels:
        y_labels = json.loads(args.y_labels)

    if args.labels:
        # If labels provided, use the old multi-log comparison
        plot_multiple_logs(args.logs, args.output, args.labels, y_labels)
    else:
        # Otherwise, compute error bars from multiple runs
        create_comparison_plot(args.logs, args.output, y_labels)


if __name__ == "__main__":
    main()
