import argparse
import json
from pathlib import Path

import matplotlib.pyplot as plt
import numpy as np
import yaml


def extract_metrics(log_data):
    results = [r for r in log_data.get("results", []) if r.get("success")]
    metrics = {
        "total": 0,
        "correct": 0,
        "incorrect": 0,
        "modification_reasoning": 0,
        "mr_when_correct": 0,
        "mr_when_incorrect": 0,
    }

    for r in results:
        # Get classification results
        classification = r.get("classification", {})
        mr_data = classification.get("modification_reasoning", {})
        correct_data = classification.get("correct_mr", {})

        # Check if both classifiers succeeded
        if not (mr_data.get("success") and correct_data.get("success")):
            continue

        metrics["total"] += 1

        # Extract values
        has_mr = mr_data.get("extracted", {}).get("final_answer") == "Yes"
        is_correct = correct_data.get("extracted", {}).get("correct") == "Yes"

        if is_correct:
            metrics["correct"] += 1
            if has_mr:
                metrics["mr_when_correct"] += 1
        else:
            metrics["incorrect"] += 1
            if has_mr:
                metrics["mr_when_incorrect"] += 1

        if has_mr:
            metrics["modification_reasoning"] += 1

    accuracy = metrics["correct"] / metrics["total"] * 100 if metrics["total"] else 0
    mr_rate = (
        metrics["modification_reasoning"] / metrics["total"] * 100
        if metrics["total"]
        else 0
    )

    # Calculate MR rates conditional on correctness
    mr_when_correct = (
        metrics["mr_when_correct"] / metrics["correct"] * 100
        if metrics["correct"]
        else 0
    )
    mr_when_incorrect = (
        metrics["mr_when_incorrect"] / metrics["incorrect"] * 100
        if metrics["incorrect"]
        else 0
    )

    return {
        "accuracy": accuracy,
        "mr_rate": mr_rate,
        "mr_when_correct": mr_when_correct * metrics["correct"] / metrics["total"]
        if metrics["total"]
        else 0,
        "mr_when_incorrect": mr_when_incorrect * metrics["incorrect"] / metrics["total"]
        if metrics["total"]
        else 0,
    }


def process_setup(config, setup_name):
    setup = config["setups"][setup_name]
    log_pairs = []

    # Check if this is a comparison setup (default vs scratchpad)
    if "log_pairs" in setup:
        log_pairs = [
            (
                Path(p.get("default", p.get("log"))),
                Path(p.get("scratchpad", p.get("log"))),
            )
            for p in setup["log_pairs"]
        ]
    elif "default_logs" in setup and "scratchpad_logs" in setup:
        log_pairs = list(
            zip(
                [Path(log) for log in setup["default_logs"]],
                [Path(log) for log in setup["scratchpad_logs"]],
            )
        )
    elif "default_log" in setup and "scratchpad_log" in setup:
        log_pairs = [(Path(setup["default_log"]), Path(setup["scratchpad_log"]))]
    elif "logs" in setup:
        # Single model analysis (no comparison)
        log_pairs = [(Path(log), Path(log)) for log in setup["logs"]]
    elif "log" in setup:
        # Single log analysis
        log_pairs = [(Path(setup["log"]), Path(setup["log"]))]
    else:
        raise ValueError(f"No valid log configuration found for setup '{setup_name}'")

    metrics_list = []
    for default_path, scratchpad_path in log_pairs:
        with open(default_path) as f:
            default_m = extract_metrics(json.load(f))
        with open(scratchpad_path) as f:
            scratchpad_m = extract_metrics(json.load(f))

        metrics_list.append(
            {
                "mr_rate": scratchpad_m["mr_rate"],
                "accuracy": scratchpad_m["accuracy"],
                "mr_when_correct": scratchpad_m["mr_when_correct"],
                "mr_when_incorrect": scratchpad_m["mr_when_incorrect"],
            }
        )

    if len(metrics_list) > 1:
        return {
            k: {
                "mean": np.mean([m[k] for m in metrics_list]),
                "std": np.std([m[k] for m in metrics_list]),
            }
            for k in metrics_list[0]
        }
    else:
        return {k: {"mean": metrics_list[0][k], "std": 0} for k in metrics_list[0]}


def plot_stacked_bars(config, output_path=None, mr_breakdown=True):
    # Keep the same serif font settings
    plt.rcParams.update(
        {
            "font.family": "serif",
            "font.serif": ["Times New Roman", "Times", "DejaVu Serif", "serif"],
        }
    )

    # Collect metrics for each setup
    data = []
    for name in config["setups"]:
        metrics = process_setup(config, name)
        setup = config["setups"][name]
        display_name = setup.get("title", name)  # Use title if provided
        data.append(
            {
                "name": display_name,
                "mr_rate": metrics["mr_rate"]["mean"],
                "mr_rate_err": metrics["mr_rate"]["std"],
                "accuracy": metrics["accuracy"]["mean"],
                "accuracy_err": metrics["accuracy"]["std"],
                "mr_when_correct": metrics["mr_when_correct"]["mean"],
                "mr_when_correct_err": metrics["mr_when_correct"]["std"],
                "mr_when_incorrect": metrics["mr_when_incorrect"]["mean"],
                "mr_when_incorrect_err": metrics["mr_when_incorrect"]["std"],
            }
        )
        print(f"\n{display_name}:")
        print(
            f"  MR rate: {metrics['mr_rate']['mean']:.1f}% ± {metrics['mr_rate']['std']:.1f}%"
        )
        print(
            f"  Accuracy: {metrics['accuracy']['mean']:.1f}% ± {metrics['accuracy']['std']:.1f}%"
        )
        print(
            f"  MR when correct: {metrics['mr_when_correct']['mean']:.1f}% ± {metrics['mr_when_correct']['std']:.1f}%"
        )
        print(
            f"  MR when incorrect: {metrics['mr_when_incorrect']['mean']:.1f}% ± {metrics['mr_when_incorrect']['std']:.1f}%"
        )

    n = len(data)

    # Figure: 1 row x 2 columns (metrics as columns)
    fig_height = 3  # scales with number of setups
    fig, axes = plt.subplots(1, 2, figsize=(2 * 3.5, fig_height), sharey=True)
    plt.subplots_adjust(wspace=0)

    # Vertical placement
    bar_height = 0.075
    step = 0.08  # vertical spacing between rows
    y_pos = [((n - 1) / 2 - i) * step for i in range(n)]
    error_kw = {"elinewidth": 0.75, "capsize": 2, "alpha": 0.75, "ecolor": "0.1"}

    # Left column: Accuracy (single metric)
    ax_acc = axes[0]
    for i, d in enumerate(data):
        ax_acc.barh(
            y_pos[i],
            d["accuracy"],
            bar_height,
            color="#20B2AA",
            alpha=0.8,
            label="Accuracy" if i == 0 else "",
            xerr=d["accuracy_err"],
            error_kw=error_kw,
        )

    # Right column: MR Rate (with optional breakdown)
    ax_mr = axes[1]
    for i, d in enumerate(data):
        if mr_breakdown:
            # Stacked bars showing MR breakdown by correctness
            ax_mr.barh(
                y_pos[i],
                d["mr_when_correct"],
                bar_height,
                color="#8B4513",
                alpha=0.8,
                label="MR when correct" if i == 0 else "",
                xerr=d["mr_when_correct_err"],
                error_kw=error_kw,
            )
            ax_mr.barh(
                y_pos[i],
                d["mr_when_incorrect"],
                bar_height,
                left=d["mr_when_correct"],
                color="#D2691E",
                alpha=0.8,
                label="MR when incorrect" if i == 0 else "",
                xerr=d["mr_when_incorrect_err"],
                error_kw=error_kw,
            )
        else:
            # Just show total MR rate
            ax_mr.barh(
                y_pos[i],
                d["mr_rate"],
                bar_height,
                color="#8B4513",
                alpha=0.8,
                label="Modification Reasoning Rate" if i == 0 else "",
                xerr=d["mr_rate_err"],
                error_kw=error_kw,
            )

    # Common formatting
    for j, ax in enumerate(axes):
        ax.set_xlim(0, 100)
        pad = 0.045
        ax.set_ylim(min(y_pos) - pad, max(y_pos) + pad)
        ax.set_yticks(y_pos)
        ax.set_yticklabels([d["name"] for d in data])
        ax.grid(True, alpha=0.1, axis="x")
        ax.set_axisbelow(True)

        # Column titles
        if j == 0:
            ax.set_title("Accuracy (%)", fontsize=11, pad=10)
        else:
            ax.set_title("Modification Reasoning Rate (%)", fontsize=11, pad=10)

        for spine in ["top", "right"] + (["left"] if j > 0 else []):
            ax.spines[spine].set_visible(False)
        if j > 0:
            ax.tick_params(left=False)

    # Legend based on mode
    from matplotlib.patches import Rectangle

    if mr_breakdown:
        colors = ["#20B2AA", "#8B4513", "#D2691E"]
        labels = ["Accuracy", "MR when correct", "MR when incorrect"]
        alphas = [0.8, 0.8, 0.8]
    else:
        colors = ["#20B2AA", "#8B4513"]
        labels = ["Accuracy", "Modification Reasoning Rate"]
        alphas = [0.8, 0.8]

    handles = [Rectangle((0, 0), 1, 1, fc=c, alpha=a) for c, a in zip(colors, alphas)]
    fig.legend(
        handles=handles,
        labels=labels,
        loc="lower center",
        ncol=len(labels),
        frameon=False,
        fontsize=11,
        bbox_to_anchor=(0.5, -0.15),
        handlelength=1,
        handleheight=1,
    )

    plt.tight_layout()
    if output_path:
        plt.savefig(output_path, dpi=300, bbox_inches="tight")
        print(f"\nPlot saved to: {output_path}")
    else:
        plt.show()


def plot_scatter(config, output_path=None):
    data = []
    for name in config["setups"]:
        metrics = process_setup(config, name)
        setup = config["setups"][name]
        display_name = setup.get("title", name)
        data.append(
            {
                "name": display_name,
                "accuracy": metrics["accuracy"]["mean"],
                "mr_rate": metrics["mr_rate"]["mean"],
            }
        )
        print(f"\n{display_name}:")
        print(f"  Accuracy: {metrics['accuracy']['mean']:.1f}%")
        print(f"  MR rate: {metrics['mr_rate']['mean']:.1f}%")

    _, ax = plt.subplots(figsize=(10, 8))

    markers = ["o", "s", "^", "D", "v", "*", "p", "h", "<", ">", "P", "X"]
    colors = [
        "#0081A7",
        "#F77F00",
        "#9D4EDD",
        "#06A77D",
        "#D62246",
        "#003049",
        "#F4A460",
        "#2E8B57",
        "#FF1493",
        "#4682B4",
        "#8B4513",
        "#DA70D6",
    ]

    for i, d in enumerate(data):
        ax.scatter(
            d["mr_rate"],
            d["accuracy"],
            s=200,
            alpha=0.8,
            c=colors[i % len(colors)],
            marker=markers[i % len(markers)],
            label=d["name"],
            edgecolors="black",
            linewidth=0.5,
        )

    ax.set_xlabel("Modification Reasoning Rate (%)", fontsize=12)
    ax.set_ylabel("Accuracy (%)", fontsize=12)
    ax.set_title(
        "Accuracy vs Modification Reasoning Rate", fontsize=14, fontweight="bold"
    )
    ax.set_xlim(0, 100)
    ax.set_ylim(0, 100)
    ax.grid(True, alpha=0.3)
    ax.legend(
        loc="upper center",
        frameon=True,
        fancybox=True,
        shadow=True,
        bbox_to_anchor=(0.5, -0.15),
        ncol=3,
        borderaxespad=0,
    )

    plt.tight_layout()
    if output_path:
        plt.savefig(output_path, dpi=300, bbox_inches="tight")
        print(f"\nPlot saved to: {output_path}")
    else:
        plt.show()


def main():
    parser = argparse.ArgumentParser(
        description="Plot modification reasoning rates and accuracy"
    )
    parser.add_argument(
        "--config",
        type=Path,
        required=True,
        help="YAML configuration file with log paths",
    )
    parser.add_argument("--output", type=Path, help="Output file path for the plot")
    parser.add_argument(
        "--scatter",
        action="store_true",
        help="Create scatter plot instead of bar chart",
    )
    parser.add_argument(
        "--no-mr-breakdown",
        action="store_true",
        help="Don't break down MR rate by correctness",
    )

    args = parser.parse_args()

    with open(args.config) as f:
        config = yaml.safe_load(f)

    if args.scatter:
        plot_scatter(config, args.output)
    else:
        plot_stacked_bars(config, args.output, mr_breakdown=not args.no_mr_breakdown)


if __name__ == "__main__":
    main()
